#define _FORCE_C_ (1)
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "vtc.h"
#include "vtclocal.h"
#include "gp5util.h"

static void get_force_none(int ni, double (*xi)[3], int nj, double (*xj)[3], double *mj,
			      double eps, double (*a)[3], double *p);
static void get_force_and_potential_host(int ni, double (*xi)[3], int nj, double (*xj)[3], double *mj,
			   double eps, double (*a)[3], double *p);
static void get_force_host(int ni, double (*xi)[3], int nj, double (*xj)[3], double *mj,
			   double eps, double (*a)[3], double *p);
static void get_potential_host(int ni, double (*xi)[3], int nj, double (*xj)[3], double *mj,
			   double eps, double (*a)[3], double *p);
static void get_force_grape(int ni, double (*xi)[3], int nj, double (*xj)[3], double *mj,
			    double eps, double (*a)[3], double *p);
static void get_potential_grape(int ni, double (*xi)[3], int nj, double (*xj)[3], double *mj,
				double eps, double (*a)[3], double *p);


void (*vtc_force_calculator[])(int ni, double (*xi)[3], int nj, double (*xj)[3], double *mj,
			       double eps, double (*a)[3], double *p) = {
				   get_force_none,          /* no force calculation */
				   get_force_and_potential_host,          /* force & potential on host */
				   get_force_host,          /* force on host */
				   get_potential_host,          /* potential on host */
				   get_force_grape,         /* force & potential on GRAPE */
				   get_force_grape,         /* force on GRAPE */
				   get_force_grape,         /* potential on GRAPE */
			       };

void
get_force_none(int ni, double (*xi)[3], int nj, double (*xj)[3], double *mj,
	       double eps, double (*a)[3], double *p)
{
    int i, k;
    static int firstcall = 1;

    if (firstcall) {
	firstcall = 0;
	fprintf(stderr, "get_force_grape DOES NOT calculate force\n");
    }
    for (i = 0; i < ni; i++) {
	for (k = 0; k < 3; k++) {
	    a[i][k] = 0.0;
	}
	p[i] = 0.0;
    }
}

void
get_force_and_potential_host(int ni, double (*xi)[3], int nj, double (*xj)[3], double *mj,
	       double eps, double (*a)[3], double *p)
{
    int i, j, k;
    double r, r2, r3;
    double eps2 = eps * eps;

    for (i = 0; i < ni; i++) {
	for (k = 0; k < 3; k++) {
	    a[i][k] = 0.0;
	}
	p[i] = 0.0;
    }
    for (i = 0; i < ni; i++) {
	for (j = 0; j < nj; j++) {
	    double dx[3];
	    r2 = eps2;
	    for (k = 0; k < 3; k++) {
		dx[k] = xj[j][k] - xi[i][k];
	    }
	    for (k = 0; k < 3; k++) {
		r2 += dx[k] * dx[k];
	    }
	    r = sqrt(r2);
	    r3 = r * r2;
	    double mjr3inv = mj[j] / r3;
	    for (k = 0; k < 3; k++) {
		a[i][k] += mjr3inv * dx[k];
	    }
	    p[i] -= mj[j] / r;
	}
    }
}

void
get_force_host(int ni, double (*xi)[3], int nj, double (*xj)[3], double *mj,
	       double eps, double (*a)[3], double *p)
{
    int i, j, k;
    double r2, r3;
    double eps2 = eps * eps;

    for (i = 0; i < ni; i++) {
	for (k = 0; k < 3; k++) {
	    a[i][k] = 0.0;
	}
	p[i] = 0.0;
    }
    for (i = 0; i < ni; i++) {
	for (j = 0; j < nj; j++) {
	    double dx[3];
	    r2 = eps2;
	    for (k = 0; k < 3; k++) {
		dx[k] = xj[j][k] - xi[i][k];
	    }
	    for (k = 0; k < 3; k++) {
		r2 += dx[k] * dx[k];
	    }
	    r3 = sqrt(r2) * r2;
	    double mjr3inv = mj[j] / r3;
	    for (k = 0; k < 3; k++) {
		a[i][k] += mjr3inv * dx[k];
	    }
	}
    }
}

void
get_potential_host(int ni, double (*xi)[3], int nj, double (*xj)[3], double *mj,
	       double eps, double (*a)[3], double *p)
{
    int i, j, k;
    double r, r2;
    double eps2 = eps*eps;

    for (i = 0; i < ni; i++) {
	for (k = 0; k < 3; k++) {
	    a[i][k] = 0.0;
	}
	p[i] = 0.0;
    }
    for (i = 0; i < ni; i++) {
	for (j = 0; j < nj; j++) {
	    double dx[3];
	    r2 = eps2;
	    for (k = 0; k < 3; k++) {
		dx[k] += (xj[j][k] - xi[i][k]);
	    }
	    for (k = 0; k < 3; k++) {
		r2 += dx[k] * dx[k];
	    }
	    r = sqrt(r2);
	    p[i] -= mj[j] / r;
	}
    }
}

#ifdef NOGRAPE

int
vtc_auto_eliminate_self_interaction(int calculator)
{
    return FALSE;
}

void
vtc_set_scale(double xscale)
{
    /* nop */
}

void
get_force_grape(int ni, double (*xi)[3], int nj, double (*xj)[3], double *mj,
		double eps, double (*a)[3], double *p)
{
    /* nop */
}

void
get_potential_grape(int ni, double (*xi)[3], int nj, double (*xj)[3], double *mj,
		double eps, double (*a)[3], double *p)
{
    /* nop */
}

#else /* !NOGRAPE */

static double holdtime = 0.0;
static int is_grape_opened = 0;
static int jmemsize = 0;

#include "g5util.h"


/* wallclock time */

/* elapsed time in real world */
static void
get_cputime(double *lap, double *split)
{
    struct timeval x;
    double sec,microsec;

    gettimeofday(&x, NULL);
    sec = x.tv_sec;
    microsec = x.tv_usec;

    *lap = sec + microsec / 1000000.0 - *split;
    *split = sec + microsec / 1000000.0;
}

double
grape_holdtime(void)
{
    return (holdtime);
}

int
grape_is_opened(void)
{
    return (is_grape_opened);
}

void
grape_open(void)
{
    double lt=0.0, st=0.0;
    if (is_grape_opened) {
	Cfprintf(stderr, "open_grape: already opened\n");
	return;
    }
    fprintf(stderr, "open GRAPE-5\n");
    g5_open();
    get_cputime(&lt, &st);
    holdtime = st;
    jmemsize = JMEMSIZE;
    is_grape_opened = 1;
}

void
grape_close(void)
{
    if (!is_grape_opened) {
	Cfprintf(stderr, "close_grape: not opened\n");
	return;
    }
    fprintf(stderr, "close GRAPE-5\n");
    g5_close();
    is_grape_opened = 0;
}

void
vtc_close_grape(void)
{
    grape_close();
}

int
vtc_auto_eliminate_self_interaction(int calculator)
{
    return FALSE;
}

void
vtc_set_scale(double xmax, double mmin)
{
    static int firstcall = 1;

    if (!grape_is_opened()) {
	grape_open();
    }
    g5_set_range(-xmax, xmax, mmin);
}

#define NIMAX (1200000)
void
get_force_grape(int ni, double (*xi)[3], int nj, double (*xj)[3], double *mj,
		double eps, double (*a)[3], double *p)
{
    static double atmp[NIMAX][3];
    static int firstcall = 1;
    int i, j, k, njj;
    int npipe = g5_get_number_of_pipelines();
    double lt=0.0, st=0.0;

    if (NIMAX < ni) {
	fprintf(stderr, "%s get_force_grape: too large ni (%d)\n", __FILE__, ni);
	exit(1);
    }
    if (!grape_is_opened()) {
	grape_open();
    }
    for (i = 0; i < ni; i++) {
	for (k = 0; k < 3; k++) {
	    a[i][k] = 0.0;
	}
    }
    for (j = 0; j < nj; j += jmemsize) {
	if (j + jmemsize > nj) {
	    njj = nj - j;
	}
	else {
	    njj = jmemsize;
	}

        // fprintf(stderr, "njj: %d\n", njj);
	// g5_set_jp(0, njj, mj + j, xj + j);
	g5_set_xmj(0, njj, xj + j, mj + j);
        g5_set_n(njj);
        // g5_set_eps2_to_all(eps*eps);
        g5_set_eps_to_all(eps);

	for (i = 0; i < ni; i += npipe) {
	    int nii;

	    if (i + npipe > ni) {
                nii = ni - i;
	    }
	    else {
		nii = npipe;
	    }
	    g5_set_xi(nii, (double (*)[3])xi[i]);
            g5_run();
	    g5_get_force(nii, (double (*)[3])atmp[i], p+i);
	}
	for (i = 0; i < ni; i++) {
	    for (k = 0; k < 3; k++) {
		a[i][k] += atmp[i][k];
	    }
	}
    }
    if (firstcall) {
	firstcall = 0;
	fprintf(stderr, "Warning: G5 does not calculate potential. The value returned is just a dummy.\n");
    }
    for (i = 0; i < ni; i++) {
	p[i] = -0.1;
    }
    get_cputime(&lt, &st);
    //    fprintf(stderr, "%f    %f\n", st, grape_holdtime());
    if (st-grape_holdtime() > 15.0) {
	grape_close();
    }

    Cfprintf(stderr, "nj: %d a: %f %f %f p: %f\n",
	     nj, a[i][0], a[i][1], a[i][2], p[i]);
}

#endif /* NOGRAPE */
