#include <stdio.h>
#include <math.h>
#include <assert.h>
#include "vtc.h"
#include "vtclocal.h"

static void get_force_direct_grape(Forceinfo *fi, Nbodyinfo *nb);
static void get_force_direct_host(Forceinfo *fi, Nbodyinfo *nb);

/* 'typical' params for direct-summation algorithm */
void
vtc_get_default_direct_params(Forceinfo *fi)
{
    fi->eps = 0.02;
    fi->calculator = HOST;
    fi->ninteraction = 0;
}

#define NIMAX (2400000)
#define NJMAX (1200000)

void vtc_get_force_direct(Forceinfo *fi, Nbodyinfo *nb)
{
    switch (fi->calculator) {
      case HOST:
      case HOST_FORCEONLY:
      case HOST_POTENTIALONLY:
	get_force_direct_host(fi, nb);
	break;
      default:
	get_force_direct_grape(fi, nb);
	break;
    }
}

static void
get_force_direct_grape(Forceinfo *fi, Nbodyinfo *nb)
{
    int i, k, nj, off;
    double epsinv;
    static double atmp[NIMAX][3], ptmp[NIMAX];

    assert(NIMAX > nb->n);

    if (fi->calculator != GRAPE_POTENTIALONLY) {
	for (i = 0; i < nb->n; i++) {
	    for (k = 0; k < 3; k++) {
		nb->a[i][k] = 0.0;
	    }
	}
    }
    if (fi->calculator != GRAPE_FORCEONLY) {
	for (i = 0; i < nb->n; i++) {
	    nb->p[i] = 0.0;
	}
    }
    off = 0;
    nj = NJMAX;
    while (off < nb->n) {
	if (off + nj > nb->n) {
	    nj = nb->n - off;
	}
	fprintf(stderr, "off: %d n: %d nj: %d\n", off, nb->n, nj);

	(vtc_force_calculator[fi->calculator])(nb->n, nb->x, nj, nb->x+off, nb->m+off, fi->eps,
					       atmp, ptmp);
	if (fi->calculator != GRAPE_POTENTIALONLY) {
	    for (i = 0; i < nb->n; i++) {
		for (k = 0; k < 3; k++) {
		    nb->a[i][k] += atmp[i][k];
		}
	    }
	}
	if (fi->calculator != GRAPE_FORCEONLY) {
	    for (i = 0; i < nb->n; i++) {
		nb->p[i] += ptmp[i];
	    }
	    if (!vtc_auto_eliminate_self_interaction(fi->calculator)) {
		if (fi->eps != 0.0) { 
		    epsinv = 1.0/fi->eps;
		    for (i = 0; i < nb->n; i++) {
			nb->p[i] += nb->m[i] * epsinv;
		    }
		}
	    }
	}
	off += nj;
    }
}


#if 1 // no potential calculation. calculate force only. 

static void
get_force_direct_host(Forceinfo *fi, Nbodyinfo *nb)
{
    int i, j, k, n;
    double r, r2, r3;
    double eps2, aij, pij, (*x)[3], *m;
    static double atmp[NIMAX][3], ptmp[NIMAX];

    n = nb->n;
    x = nb->x;
    m = nb->m;
    assert(NIMAX > n);
    assert(NJMAX > n);

    for (i = 0; i < nb->n; i++) {
	for (k = 0; k < 3; k++) {
	    atmp[i][k] = 0.0;
	}
	ptmp[i] = 0.0;
    }

    eps2 = fi->eps * fi->eps;
    for (i = 0; i < n; i++) {
	for (j = i + 1; j < n; j++) {
	    double dx[3];
	    r2 = eps2;
	    for (k = 0; k < 3; k++) {
		dx[k] = x[j][k] - x[i][k];
	    }
	    for (k = 0; k < 3; k++) {
		r2 += dx[k] * dx[k];
	    }
	    r  = sqrt(r2);
	    r3 = r * r2;
	    double mr3inv = m[j] / r3;
	    for (k = 0; k < 3; k++) {
		aij = mr3inv * dx[k];
		atmp[i][k] += aij;
		atmp[j][k] -= aij;
	    }
	}
    }
    for (i = 0; i < nb->n; i++) {
	for (k = 0; k < 3; k++) {
	    nb->a[i][k] = atmp[i][k];
	}
    }
}

#else // calculate force and potential.

static void
get_force_direct_host(Forceinfo *fi, Nbodyinfo *nb)
{
    int i, j, k, n;
    double r, r2, r3;
    double eps2, aij, pij, (*x)[3], *m;
    static double atmp[NIMAX][3], ptmp[NIMAX];

    n = nb->n;
    x = nb->x;
    m = nb->m;
    assert(NIMAX > n);
    assert(NJMAX > n);

    for (i = 0; i < nb->n; i++) {
	for (k = 0; k < 3; k++) {
	    atmp[i][k] = 0.0;
	}
	ptmp[i] = 0.0;
    }

    eps2 = fi->eps * fi->eps;
    for (i = 0; i < n; i++) {
	for (j = i + 1; j < n; j++) {
	    double dx[3];
	    r2 = eps2;
	    for (k = 0; k < 3; k++) {
		dx[k] = x[j][k] - x[i][k];
	    }
	    for (k = 0; k < 3; k++) {
		r2 += dx[k] * dx[k];
	    }
	    r  = sqrt(r2);
	    r3 = r * r2;
	    double mr3inv = m[j] / r3;
	    for (k = 0; k < 3; k++) {
		aij = mr3inv * dx[k];
		atmp[i][k] += aij;
		atmp[j][k] -= aij;
	    }
	    pij = m[j] / r;
	    ptmp[i] -= pij;
	    ptmp[j] -= pij;
	}
    }
    for (i = 0; i < nb->n; i++) {
	for (k = 0; k < 3; k++) {
	    nb->a[i][k] = atmp[i][k];
	}
    }
    for (i = 0; i < nb->n; i++) {
	nb->p[i] = ptmp[i];
    }
}
#endif
