#include <stdio.h>
#include <stdlib.h>
#include "g5util.h"
#include "direct.h"

/*
 * calc_gravity3(): yet another force calculator. A user can handle
 * each individual GRAPE-7 card by passing device ID (a unique integer
 * given to each GRAPE-7 card) to G5 "primitive" functions. See G5
 * reference manual for the detail.
 */
void
calc_gravity3(int id, double *mj, double (*xj)[3], double (*vj)[3],
              double eps, double (*a)[3], double *p, int n)
{
    int off, ni, np;

    g5_set_jpMC(id, 0, n, mj, xj);
    g5_set_eps_to_allMC(id, eps);
    g5_set_nMC(id, n);

    ni = g5_get_number_of_pipelines();
    for (off = 0; off < n; off += ni) {
        if (off + ni > n) {
            ni = n - off;
        }

        g5_set_xiMC(id, ni, (double (*)[3])xj[off]);
        g5_runMC(id);
        g5_get_forceMC(id, ni, (double (*)[3])a[off], &p[off]);
    }
}

int
main(int argc, char **argv)
{
    static double mj[NMAX], xj[NMAX][3], vj[NMAX][3];
    static double a[NMAX][3], p[NMAX];
    double xmax, xmin, mmin;
    double time, dt, endt;;
    double eps;
    double e, e0, ke, pe;
    int i, j, n, id;
    int nstep, step;

    eps = 0.02;
    dt = 0.01;
    endt = 1.0;
    time = 0.0;
    nstep = endt/dt;
    xmax = 64.0;
    xmin = -64.0;

    if (argc < 4) {
        fprintf(stderr, "usage: %s <infile> <outfile> <device_ID>\n",  argv[0]);
        exit(1);
    }
    id = atoi(argv[3]);
    if (id < 0 || id >= g5_get_number_of_cards()) {
        fprintf(stderr, "device_ID:%d -- out of range.\n", id);
        exit(2);
    }
    printf("Use GRAPE-7[%d]\n", id);
  
    readnbody(&n, mj, xj, vj, argv[1]);
    mmin = mj[0];
    g5_openMC(id);
    g5_set_rangeMC(id, xmin, xmax, mmin);
    calc_gravity3(id, mj, xj, vj, eps, a, p, n);
    energy(mj, vj, p, n, &ke, &pe);
    e0 = ke+pe;
    printf("ke: %f\n", ke);
    for (step = 1; step < nstep; step++) {
        push_velocity(vj, a, 0.5*dt, n);
        push_position(xj, vj, a, dt, n);
        time = time + dt;
        calc_gravity3(id, mj, xj, vj, eps, a, p, n);
        push_velocity(vj, a, 0.5*dt, n);
        if (step % (nstep/10) == 0) {
            energy(mj, vj, p, n, &ke, &pe);
            e = ke+pe;
            printf("step: %d time: %e\n", step, time);
            printf("e: %e de: %e\n", e, e-e0);
            printf("ke: %e pe: %e\n", ke, pe);
            printf("ke/pe: %e\n\n", ke/pe);
        }
    }
    g5_closeMC(id);
    writenbody(n, mj, xj, vj, argv[2]);
}
