#include <stdio.h>
#include <stdlib.h>
#include "g5util.h"
#include "direct.h"

/*
 * calc_gravity(): a force calculator. The simplest one.
 */
void
calc_gravity(double *mj, double (*xj)[3], double (*vj)[3],
	     double eps, double (*a)[3], double *p, int n)
{
    g5_set_eps2_to_all(eps*eps);
    g5_set_jp(0, n, mj, xj);
    g5_set_n(n);
    g5_calculate_force_on_x(xj, a, p, n);

    if (eps != 0.0) {
        int i;
        double epsinv;
        epsinv = 1.0 / eps;
        for (i = 0; i < n; i++) {
            p[i] += mj[i] * epsinv;
        }
    }

}

/*
 * calc_gravity2(): another force calculator. A bit more complex than
 * calc_gravity(), but much more flexible.
 */
void
calc_gravity2(double *mj, double (*xj)[3], double (*vj)[3],
              double eps, double (*a)[3], double *p, int n)
{
    int off, ni, np;

    g5_set_jp(0, n, mj, xj);
    g5_set_eps2_to_all(eps*eps);
    g5_set_n(n);

    ni = g5_get_number_of_pipelines();
    for (off = 0; off < n; off += ni) {
        if (off + ni > n) {
            ni = n - off;
        }

        g5_set_xi(ni, (double (*)[3])xj[off]);
        g5_run();
        g5_get_force(ni, (double (*)[3])a[off], &p[off]);
    }
    if (eps != 0.0) {
        int i;
        double epsinv;
        epsinv = 1.0 / eps;
        for (i = 0; i < n; i++) {
            p[i] += mj[i] * epsinv;
        }
    }
}

int
main(int argc, char **argv)
{
    static double mj[NMAX], xj[NMAX][3], vj[NMAX][3];
    static double a[NMAX][3], p[NMAX];
    double xmax, xmin, mmin;
    double time, dt, endt;;
    double eps;
    double e, e0, ke, pe;
    int i, j, n;
    int nstep, step;

    eps = 0.02;
    dt = 0.01;
    endt = 1.0;
    time = 0.0;
    nstep = endt/dt;
    xmax = 64.0;
    xmin = -64.0;

    if (argc < 3) {
        fprintf(stderr, "usage: %s <infile> <outfile>\n",  argv[0]);
        exit(1);
    }
  
    readnbody(&n, mj, xj, vj, argv[1]);
    mmin = mj[0];
    g5_open();
    g5_set_range(xmin, xmax, mmin);
    calc_gravity(mj, xj, vj, eps, a, p, n);
    energy(mj, vj, p, n, &ke, &pe);
    e0 = ke+pe;
    printf("ke: %f  pe: %f  e0: %f\n", ke, pe, e0);
    for (step = 1; step < nstep; step++) {
        push_velocity(vj, a, 0.5*dt, n);
        push_position(xj, vj, a, dt, n);
        time = time + dt;
        calc_gravity(mj, xj, vj, eps, a, p, n);
        push_velocity(vj, a, 0.5*dt, n);
#ifdef ANIM
        plot_star(xj, n, time, 0.3, mj, mj[0]);
#endif /* ANIM */

        if (step % (nstep/10) == 0) {
            energy(mj, vj, p, n, &ke, &pe);
            e = ke+pe;
            printf("step: %d time: %e\n", step, time);
            printf("e: %e de: %e\n", e, e-e0);
            printf("ke: %e pe: %e\n", ke, pe);
            printf("ke/pe: %e\n\n", ke/pe);
        }
    }
    g5_close();
    writenbody(n, mj, xj, vj, argv[2]);
}
