#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include "hibutil.h"

static char* Productname[] = {
    "",
    "GRAPE-7",  // 1
    "GRAPE-7E", // 2
    "GRAPE-DR", // 3
    "GRAPE-9",  // 4
};
static char* Modelname[][16] = {
    //    1                     2            3           4             5            6            7            8
    {},
    { "", "model100"          , "model300D", "model300", ""          , ""         , "model600" , ""         , "model800" , },
    { "", "model100"          , ""         , ""        , ""          , ""         , ""         , ""         , ""         , },
    { "", "TB1"               , "TB2"      , "model460", "model2000" , ""         , ""         , ""         , ""         , },
    { "", "model800/1600"     , ""         , ""        , ""          , ""         , ""         , ""         , ""         , },
};

static char* Backendname[] = {
    "empty",
    "G5   ",  // 1
    "G6   ",  // 2
    "GMP4 ",  // 3
    "GMP6 ",  // 4
    "GMP8 ",  // 5
    "G5nb ",  // 6  G5 + nb, pot
    "G6nb ",  // 7  G6 + nb, nnb, pot, cutoff
};

static int Productid, Modelid, Backendid, Revision;


void g5info(UINT32 binfo);
void g6info(UINT32 binfo);
void g6nbinfo(UINT32 binfo);
void noinfo(UINT32 binfo);

static void (*longinfo[])(UINT32 binfo) = {
    noinfo,
    g5info, // G5
    g6info, // G6
    noinfo, // GMP4
    noinfo, // GMP5
    noinfo, // GMP6
    noinfo, // G5nb
    g6nbinfo, // G6nb
};


static unsigned int G7modelid2nchip[] = {
    0,
    1, // m100
    3, // m300d
    3, // m300
    0,
    0,
    6, // m600
    0,
    1, // m800
};

void
g5info(UINT32 binfo)
{
    int npipe    =   (binfo >>  0) & 0xff;
    int jmem     = 2048 << ((binfo>>8) & 0x3);
    int p3m      =   (binfo >> 15) & 0x1;
    int nb       =   (binfo >> 16) & 0x1;
    int foutfifo = (((binfo >> 10) & 0x3) << 10);
    int eps2     =   (binfo >> 14) & 0x1;
    int modelid  = (binfo >> 24) & 0xf;
    int nchip    = G7modelid2nchip[modelid];

    printf("        number of chips             : %d\n", nchip);
    printf("        number of pipelines/chip    : %d\n", npipe);
    printf("        j-particle memory size/chip : %d particles\n", jmem);
    printf("        P3M cutoff                  : %s\n", p3m ?  "available"      : "not available");
    printf("        neighbor search             : %s\n", nb  ?  "available"      : "not available");
    printf("        fout fifo size              : %d bytes\n", foutfifo * sizeof(UINT64));
    printf("        number format               : %s\n", eps2 ? "floating-point" : "logarithmic");
}

void
g6info(UINT32 binfo)
{
    int npipe    = (binfo >>  0) & 0xff;
    int jmem;
    switch (Productid) {
      case 4: // GRAPE-9
        switch (Revision) {
          case 0:
            jmem = 2048 << (unsigned int)((binfo>>8) & 0xf);
            break;
          case 1:
            jmem = 2048;
            break;
          default:
            fprintf(stderr, "revision %d not supported.\n", Revision);
            exit(1);
        }
        printf("        number of pipelines/chip    : %d\n", npipe);
        printf("        j-particle memory size/chip : %d particles\n", jmem);
        break;
      default:
        noinfo(0);
    }
}

void
g6nbinfo(UINT32 binfo)
{
    int npipe    = (binfo >>  0) & 0xff;
    int jmem;
    switch (Productid) {
      case 4: // GRAPE-9
        switch (Revision) {
          case 0:
            jmem = 2048 << (unsigned int)((binfo>>8) & 0xf);
            break;
          case 1:
            jmem = 2048;
            break;
          default:
            fprintf(stderr, "revision %d not supported.\n", Revision);
            exit(1);
        }
        printf("        number of pipelines/chip    : %d\n", npipe);
        printf("        j-particle memory size/chip : %d particles\n", jmem);
        break;
      default:
        noinfo(0);
    }
}


void
noinfo(UINT32 binfo)
{
    printf("        no verbose info available.\n");
}

static void
showusage(char *cmdname)
{
    fprintf(stderr, "'%s' shows GRAPE devices available.\n", cmdname);
    fprintf(stderr, "usage: %s [options]\n", cmdname);
    fprintf(stderr, "    -l            be verbose.\n");
    fprintf(stderr, "    -v            same as -l.\n");
    fprintf(stderr, "    -d num        print num-th device only\n"
	            "                  (print all devices by default).\n");
    fprintf(stderr, "    -h            print this message.\n");
}

int
main(int argc, char **argv)
{
    Hib *h;
    UINT32 binfo;
    int c, ic;
    char buf[256];
    int devid = -1;
    char* param = "d:lhv";
    static int is_long_format = 0;

    while ((c = getopt(argc, argv, param)) != EOF) {
	switch (c) {
	  case 'l':
	  case 'v':
	    is_long_format = 1;
	    break;
	  case 'd':
	    devid = atoi(optarg);
	    if (devid < 0 || hib_ndevice() <= devid) {
		fprintf(stderr, "invalid device ID (%d)\n", devid);
		exit(1);
	    }
	    break;
	  case 'h':
	    showusage(argv[0]);
	    exit(0);
	    break;
	}	
    }

    printf("devid grape(model)              backend-logic\n");

    hib_set_warn_level(1);
    for (ic = 0; ic < hib_ndevice(); ic++) {

        // print info for only one device if a devid is given.
	if (devid != -1 && devid != ic) continue;

        h = hib_try_openMC(ic);
        if (h) { // successfully opened.
            binfo = hib_mem_readMC(ic, h->r->boardinfo);
            Productid = (binfo >> 28) & 0xf;
            Modelid = (binfo >> 24) & 0xf;
            Backendid = (binfo >> 20) & 0xf;
            Revision  = (binfo >> 16) & 0xf;
            hib_closeMC(ic);

            sprintf(buf, "%s(%s rev%d)",
                    Productname[Productid], Modelname[Productid][Modelid], Revision);
            printf("% 2d    %-25s %s", ic, buf, Backendname[Backendid]);
            printf("\n");
            if (is_long_format) {
                (*longinfo[Backendid])(binfo);
            }
        }
        else { // open failed. locked by another process.
            printf("% 2d    device%d locked by another process (pid:%d). ",
                   ic, ic, hiberr.locked_by_pid);
            printf("device info temporary not available.\n");
        }
    }

#if 0
    int i, j;
    for (i = 0; i < 4; i++) {
	for (j = 0; j < 16; j++) {
	    printf("%s  ", Modelname[i][j]);
	}
	printf("\n");
    }
#endif
}
