/*
 * GRAPE-5 standard functions
 */

void g5_open(void)
{
    int ic;

    init_envs();
    set_resend_flags();
    for (ic = 0; ic < hib_ndevice(); ic++) {
	if (Device[ic] == 0) continue;
	g5_openMC(ic);
    }
}

void g5_close(void)
{
    int ic;

    for (ic = 0; ic < hib_ndevice(); ic++) {
	if (Device[ic] == 0) continue;
	g5_closeMC(ic);
    }
}

void g5_set_range(double xmin, double xmax, double mmin)
{
    int ic;

    for (ic = 0; ic < hib_ndevice(); ic++) {
	if (Device[ic] == 0) continue;
	g5_set_rangeMC(ic, xmin, xmax, mmin);
    }
}

int g5_get_jmemsize(void)
{
    int ic;
    int jms = 0;

    for (ic = 0; ic < hib_ndevice(); ic++) {
	if (Device[ic] == 0) continue;
	jms += g5_get_jmemsizeMC(ic);
    }
    WARN(4, "g5_get_jmemsize returning value: %d\n", jms);
    return jms;
}

int g5_get_number_of_pipelines(void)
{
    int ic, n;
    int nmin = 65536; // any large number will do.

    // returns the smallest one
    for (ic = 0; ic < hib_ndevice(); ic++) {
	if (Device[ic] == 0) continue;
	n = g5_get_number_of_pipelinesMC(ic);
	if (nmin > n) {
	    nmin = n;
	}
    }

    WARN(5, "g5_get_number_of_pipelines returning value: %d\n", nmin);
    return nmin;
}

void g5_set_n(int n)
{
    int ic, njj, ndev;

    for (ic = 0, ndev = 0; ic < hib_ndevice(); ic++) {
	if (Device[ic] == 0) continue;

        njj = n / Ndevice;
        if (ndev < n % Ndevice) {
	  //            njj + 1;
            njj = njj + 1;	    
        }
	g5_set_nMC(ic, njj);
        ndev++;
    }
}

void g5_set_eps2_to_all(double eps2)
{
    int ic;

    for (ic = 0; ic < hib_ndevice(); ic++) {
	if (Device[ic] == 0) continue;
	g5_set_eps2_to_allMC(ic, eps2);
    }
}

void g5_set_eps2(int ni, double *eps2)
{
    int ic;

    for (ic = 0; ic < hib_ndevice(); ic++) {
	if (Device[ic] == 0) continue;
	g5_set_eps2MC(ic, ni, eps2);
    }
}

void
g5_set_eta(double eta)
{
  int ic;

  for (ic = 0; ic < hib_ndevice(); ic++) {
    if (Device[ic] == 0) continue;
    g5_set_etaMC(ic, eta);
  }
}

void
g5_set_cutoff_table(double (*ffunc)(double), double fcut, double fcor,
                    double (*pfunc)(double), double pcut, double pcor)
{
  int ic;

  for (ic = 0; ic < hib_ndevice(); ic++) {
    if (Device[ic] == 0) continue;
    g5_set_cutoff_tableMC(ic, ffunc, fcut, fcor, pfunc, pcut, pcor);
  }
}

void g5_set_jp(int adr, int nj, double *m, double (*x)[3])
{
    int ic, j, k, ndev;
    int adr0[NHIBMAX];
    int nj0[NHIBMAX];
    static double *m0[NHIBMAX];
    static double (*x0[NHIBMAX])[3];
    static int firstcall = 1;

    // allocate jp buffers.
    if (firstcall) {
        firstcall = 0; 
        for (ic = 0, ndev = 0; ic < hib_ndevice(); ic++) {
            if (Device[ic] == 0) continue;
            m0[ndev] = (double *)malloc(sizeof(double) * g5_get_jmemsizeMC(ic));
            x0[ndev] = (double (*)[3])malloc(sizeof(double) * 3 * g5_get_jmemsizeMC(ic));
            if (m0[ndev] == NULL || x0[ndev] == NULL) {
                perror("g5_set_jp");
                exit(1);
            }
            ndev++;
        }
    }

    // split m & x into jp buffers.
    for (ic = 0; ic < hib_ndevice(); ic++) {
        adr0[ic] = -1;
        nj0[ic] = 0;
    }
    for (j = 0; j < nj; j++) {
        int ndevj = (adr + j) % Ndevice;
        int adrj  = (adr + j) / Ndevice;
        if (adr0[ndevj] == -1) { // top addres is not set.
            adr0[ndevj] = adrj;
        }
        m0[ndevj][nj0[ndevj]] = m[j];
        for (k = 0; k < 3; k++) {
            x0[ndevj][nj0[ndevj]][k] = x[j][k];
        }
        nj0[ndevj]++;
    }

    // send contents of each jp buffer to corresponding device.
    for (ic = 0, ndev = 0; ic < hib_ndevice(); ic++) {
	if (Device[ic] == 0) continue;
        g5_set_jpMC(ic, adr0[ndev], nj0[ndev], m0[ndev], x0[ndev]);
        ndev++;
    }
}
 
void g5_set_xi(int ni, double (*x)[3])
{
    int ic;

    for (ic = 0; ic < hib_ndevice(); ic++) {
	if (Device[ic] == 0) continue;
	g5_set_xiMC(ic, ni, x);
    }
}

void g5_run(void)
{
    int ic;

    for (ic = 0; ic < hib_ndevice(); ic++) {
	if (Device[ic] == 0) continue;
	g5_runMC(ic);
    }
}

void g5_get_force(int ni, double (*a)[3], double *pot)
{
    int ic, i, k;
    static double atmp[NFOMAX][3];
    static double ptmp[NFOMAX];

    for (i = 0; i < ni; i++) {
        for (k = 0; k < 3; k++) {
            a[i][k] = 0.0;
        }
        pot[i] = 0.0;
    }
    for (ic = 0; ic < hib_ndevice(); ic++) {
	if (Device[ic] == 0) continue;
        g5_get_forceMC(ic, ni, atmp, ptmp);
        for (i = 0; i < ni; i++) {
            for (k = 0; k < 3; k++) {
                a[i][k] += atmp[i][k];
            }
            pot[i] += ptmp[i];
        }
    }
}

void
g5_calculate_force_on_x(double (*x)[3], double (*a)[3], double *p, int ni)
{
    int off, nii, np;

    np = g5_get_number_of_pipelines();

    for (off = 0; off < ni; off += np) {
        nii = np;
        if (off+nii > ni) {
            nii = ni - off;
        }

        g5_set_xi(nii, (double (*)[3])x[off]);
        g5_run();
        g5_get_force(nii, (double (*)[3])a[off], &p[off]);
    }
}

/************** nnb function **********************/

void g5n_set_jp(int adr, int nj, double *m, double (*x)[3], int *index)
{
    int ic, j, k, ndev;
    int adr0[NHIBMAX];
    int nj0[NHIBMAX];
    static double *m0[NHIBMAX];
    static double (*x0[NHIBMAX])[3];
    static int *index0[NHIBMAX];
    static int firstcall = 1;

    // allocate jp buffers.
    if (firstcall) {
        firstcall = 0; 
        for (ic = 0, ndev = 0; ic < hib_ndevice(); ic++) {
            if (Device[ic] == 0) continue;
            m0[ndev] = (double *)malloc(sizeof(double) * g5_get_jmemsizeMC(ic));
            index0[ndev] = (int *)malloc(sizeof(int) * g5_get_jmemsizeMC(ic));	    
            x0[ndev] = (double (*)[3])malloc(sizeof(double) * 3 * g5_get_jmemsizeMC(ic));
            if (m0[ndev] == NULL || x0[ndev] == NULL) {
                perror("g5_set_jp");
                exit(1);
            }
            ndev++;
        }
    }

    // split m & x into jp buffers.
    for (ic = 0; ic < hib_ndevice(); ic++) {
        adr0[ic] = -1;
        nj0[ic] = 0;
    }
    for (j = 0; j < nj; j++) {
        int ndevj = (adr + j) % Ndevice;
        int adrj  = (adr + j) / Ndevice;
        if (adr0[ndevj] == -1) { // top addres is not set.
            adr0[ndevj] = adrj;
        }
        m0[ndevj][nj0[ndevj]] = m[j];
        index0[ndevj][nj0[ndevj]] = index[j];	
        for (k = 0; k < 3; k++) {
            x0[ndevj][nj0[ndevj]][k] = x[j][k];
        }
        nj0[ndevj]++;
    }

    // send contents of each jp buffer to corresponding device.
    for (ic = 0, ndev = 0; ic < hib_ndevice(); ic++) {
	if (Device[ic] == 0) continue;
        g5n_set_jpMC(ic, adr0[ndev], nj0[ndev], m0[ndev], x0[ndev],index0[ndev]);
        ndev++;
    }
}
 
void g5_set_index(int ni, int *index)
{
    int ic;

    for (ic = 0; ic < hib_ndevice(); ic++) {
	if (Device[ic] == 0) continue;
	g5_set_indexMC(ic, ni, index);
    }
}

void g5n_get_force(int ni, double (*a)[3], double *pot,
                     double * rnnb2, int * innb)
{
    int ic, i, k;
    static double atmp[NFOMAX][3];
    static double ptmp[NFOMAX];
    static double rntmp[NFOMAX];
    static int intmp[NFOMAX];    

    for (i = 0; i < ni; i++) {
        for (k = 0; k < 3; k++) {
            a[i][k] = 0.0;
        }
        pot[i] = 0.0;
	rnnb2[i] = -1.0;
    }
    for (ic = 0; ic < hib_ndevice(); ic++) {
	if (Device[ic] == 0) continue;
        g5n_get_forceMC(ic, ni, atmp, ptmp, rntmp, intmp);
        for (i = 0; i < ni; i++) {
            for (k = 0; k < 3; k++) {
              a[i][k] += atmp[i][k];
            }
            pot[i] += ptmp[i];

	    if(rnnb2[i]<0.0 || rnnb2[i]>rntmp[i]) {
  	      rnnb2[i] = rntmp[i];
	      innb[i] = intmp[i];	    
	    }
        }
    }
}

void g5n_calculate_force_on_x(double (*x)[3],
				int * index,
				double (*a)[3], double *p,
				double * rnnb2,
				int * innb, int ni)
{
  int off, nii, np;

  np = g5_get_number_of_pipelines();

  for (off = 0; off < ni; off += np) {
    nii = np;
    if (off+nii > ni) {
      nii = ni - off;
    }

    g5_set_xi(nii, (double (*)[3])x[off]);
    g5_set_index(nii, index+off);
    g5_run();
    g5n_get_force(nii, (double (*)[3])a[off], &p[off], &rnnb2[off], &innb[off]);
  }
}

