#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/time.h>
#include <time.h>
#include <math.h>
#include "pcixmem.h"
#include "memmap.h"
#include "xhibutil.h"

void showstatus(int argc, char **argv);
void stopdmaw(int argc, char **argv);
void clearfifo(int argc, char **argv);
void showdmastatus(int argc, char **argv);
void configread(int argc, char **argv);
void configwrite(int argc, char **argv);
void regread(int argc, char **argv);
void regwrite(int argc, char **argv);
void pioread(int argc, char **argv);
void piowrite(int argc, char **argv);
void dmatest(int argc, char **argv);
void dmaperf(int argc, char **argv);
void dmawperf(int argc, char **argv);
void dmarperf(int argc, char **argv);
void rawperf(int argc, char **argv);
void showusage(int argc, char **argv);
void get_cputime(double *laptime, double *sprittime);

#define JMEMSIZE (32768)
static UINT64 *rbuf;
static UINT64 *wbuf;
static UINT64 *backend[NPCIXMEM];
static double posbuf[JMEMSIZE][3];
static UINT64 piowbuf[XHIBRAMWORDS];

typedef struct
{
    void (*func)();
    char *usage;
} TestMode;

static TestMode testmode[] =
    {
	showstatus, "show contents of config & XHIB-local registers",
	stopdmaw, "reset DMA and FIFO",
	clearfifo, "clear XHIB-internal FIFO",
	showdmastatus, "show DMA status",
	configread, "read config register <addr>",
	configwrite, "write config register <addr> <val>",
	regread, "read XHIB local registers mapped to BAR0 <addr>",
	regwrite, "write XHIB local registers mapped to BAR0 <addr> <val>",
	pioread, "read backend memory space mapped to BAR1 <addr>",
	piowrite, "write backend memory space mapped to BAR1 <addr> <val>",
	dmatest, "check DMA read/write function <size> (host <-> XHIB)",
	dmaperf, "measure DMA performance (host <-> XHIB)",
	dmawperf, "measure DMA write performance (host <- EHIB; bypass internal FIFO)",
	dmarperf, "measure DMA read performance (host -> EHIB; bypass internal FIFO)",
	rawperf, "raw PIO r/w & DMA r/w",
    };

int
main(int argc, char **argv)
{
    int mode;

    if (argc < 2) {
	showusage(argc, argv);
	exit (1);
    }
    mode = atoi(argv[1]);
    if (0 <= mode && mode < sizeof(testmode)/sizeof(testmode[0])) {
	testmode[mode].func(argc, argv);
    }
    else {
	showusage(argc, argv);
	exit (1);
    }
    exit (0);
}

void
showusage(int argc, char **argv)
{
    int i;
    int nitems = sizeof(testmode)/sizeof(testmode[0]);

    fprintf(stderr, "usage: %s <test_program_ID>\n", argv[0]);
    for (i = 0; i < nitems; i++) {
	fprintf(stderr, "%d) %s\n", i, testmode[i].usage);
    }
}


void
dmatest(int argc, char **argv)
{
    int devid = 0;
    int i, j, ntry, nng, size0, off;
    int size; /* in 32-bit words */
    double lt = 0.0, st = 0.0;

    if (argc < 3) {
	showusage(argc, argv);
	exit(1);
    }

    if (argc > 3) {
	devid = atoi(argv[3]);
	if (NPCIXMEM < devid+1) {
	    fprintf(stderr,
		    "too large devid(= %d).\n",
		    devid);
	    exit(1);
	}
    }

    size = atoi(argv[2]);
    printf("size %d\n", size);
    if (size < 0 || size > XHIBRAMWORDS) {
	fprintf(stderr, "inappropriate size %d\n", size);
	exit(1);
    }

    xhib_set_nclusters(NPCIXMEM);
    xhib_openMC(devid, &rbuf, &wbuf, &backend[devid]);
    xhib_set_test_modeMC(devid, TESTMODE_REFDESIGN_FIFO);

#if 0
    printf("DMA read (host -> XHIB), then write (host <- XHIB)\n");
#else
    printf("PIO write (host -> XHIB), then DMA write (host <- EHIB)\n");
    xhib_set_sendfuncMC(devid, SENDFUNC_PIOW);
    rbuf = piowbuf;
#endif

    printf("clear DMA buf...\n");

    srand48(time(NULL));
    for (i = 0; i < size+10; i++) {
	rbuf[i] = 0x123456789abc0000ll|i;
        //	rbuf[i] = lrand48() << 32 | lrand48();;
	wbuf[i] = 0xfedcba9876540000ll|i;
	//      fprintf(stderr, "rbuf[0x%02x]: 0x%016llx\n", i, rbuf[i]);
    }

    printf("DMA read size: %d words (%d bytes)\n", size, size*8);
    srand48(time(NULL));
    printf("will dmar...\n");

    xhib_sendMC(devid, size, rbuf);
    xhib_recvMC(devid, size, wbuf);

    for (nng = 0, i = 0; i < size+2; i++) {
	fprintf(stdout, "rbuf[%04d]: 0x%016llx  wbuf[%04d]: 0x%016llx",
		i, rbuf[i], i, wbuf[i]);
	if (wbuf[i] != rbuf[i] && i < size) {
	    nng++;
	    fprintf(stdout, " NG\n");
	}
	else {
	    fprintf(stdout, " \n");
	}
	if (i+1 == size) {
	    fprintf(stdout, "---- transfer size reached ----\n");
	}
    }
    printf("done\n %d words (%d bytes).\n", size, size*8);
    if (nng) {
	fprintf(stderr, "NG %d words\n", nng);
    }
    else {
	fprintf(stderr, "OK\n");
    }
    for (i = 0; i < size; i++) {
	rbuf[i] = 0;
	wbuf[i] = 0;
    }
    xhib_set_test_modeMC(devid, TESTMODE_NONE);
    xhib_closeMC(devid);
}

void
dmaperf(int argc, char **argv)
{
    int devid = 0;
    int i, j, ntry, nng, size0, off;
    int size; /* in 32-bit words */
    double lt = 0.0, st = 0.0;

    if (argc < 2) {
	showusage(argc, argv);
	exit(1);
    }
    size = atoi(argv[1]);
    printf("size: %d\n", size);

    if (argc > 2) {
	devid = atoi(argv[2]);
	if (NPCIXMEM < devid+1) {
	    fprintf(stderr,
		    "too large devid(= %d).\n",
		    devid);
	    exit(1);
	}
    }

    printf("DMA read (host -> XHIB)\n");

    xhib_set_nclusters(NPCIXMEM);
    xhib_openMC(devid, &rbuf, &wbuf, &backend[devid]);
    xhib_set_test_modeMC(devid, TESTMODE_REFDESIGN_FIFO);

    printf("clear DMA buf...\n");

    for (i = 0; i < size+10; i++) {
	rbuf[i] = 0x123456789abc0000ll|i;
	wbuf[i] = 0xfedcba9876540000ll|i;
	//      fprintf(stderr, "rbuf[0x%02x]: 0x%016llx\n", i, rbuf[i]);
    }

#define NLOOP (1e8)

    for (size = 128; size <= 512; size *= 2) {
	for (ntry = 0; ntry < 1; ntry++) {
	    get_cputime(&lt, &st);
	    for (j = 0; j < NLOOP/size; j++) {
		xhib_sendMC(devid, size, rbuf);
		xhib_recvMC(devid, size, wbuf);
	    }
	    get_cputime(&lt, &st);
	    printf("size: %d DMA read -> write: %f sec  %f MB/s\n",
		   size*8, lt, 2*sizeof(UINT64)*NLOOP/1e6/lt);
	}
    }
    xhib_set_test_modeMC(devid, TESTMODE_NONE);
    xhib_closeMC(devid);
}


void
dmawperf(int argc, char **argv)
{
    int devid = 0;
    int i, j, ntry, nng, size0, off;
    int size; /* in 32-bit words */
    double lt = 0.0, st = 0.0;

    if (argc < 2) {
	showusage(argc, argv);
	exit(1);
    }
    size = atoi(argv[1]);
    printf("size: %d\n", size);

    if (argc > 2) {
	devid = atoi(argv[2]);
	if (NPCIXMEM < devid+1) {
	    fprintf(stderr,
		    "too large devid(= %d).\n",
		    devid);
	    exit(1);
	}
    }

    printf("DMA write (host <- XHIB)\n");

    xhib_set_nclusters(NPCIXMEM);
    xhib_openMC(devid, &rbuf, &wbuf, &backend[devid]);
    xhib_set_test_modeMC(devid, TESTMODE_REFDESIGN_RAM);

    printf("clear DMA buf...\n");

    for (i = 0; i < size+10; i++) {
	rbuf[i] = 0x123456789abc0000ll|i;
	wbuf[i] = 0xfedcba9876540000ll|i;
	//      fprintf(stderr, "rbuf[0x%02x]: 0x%016llx\n", i, rbuf[i]);
    }

#define NLOOP (1e8)

    for (size = 64; size <= 4096; size *= 2) {
	for (ntry = 0; ntry < 1; ntry++) {
	    get_cputime(&lt, &st);
	    for (j = 0; j < NLOOP/size; j++) {
		xhib_recvMC(devid, size, rbuf);
	    }
	    get_cputime(&lt, &st);
	    printf("size: %d DMA write: %f sec  %f MB/s\n",
		   size*8, lt, sizeof(UINT64)*NLOOP/1e6/lt);
	}
    }
    xhib_set_test_modeMC(devid, TESTMODE_NONE);
    xhib_closeMC(devid);
}


void
dmarperf(int argc, char **argv)
{
    int devid = 0;
    int i, j, ntry, nng, size0, off;
    int size; /* in 32-bit words */
    double lt = 0.0, st = 0.0;

    if (argc < 2) {
	showusage(argc, argv);
	exit(1);
    }
    size = atoi(argv[1]);
    printf("size: %d\n", size);

    if (argc > 2) {
	devid = atoi(argv[2]);
	if (NPCIXMEM < devid+1) {
	    fprintf(stderr,
		    "too large devid(= %d).\n",
		    devid);
	    exit(1);
	}
    }

    printf("DMA read (host -> XHIB)\n");

    xhib_set_nclusters(NPCIXMEM);
    xhib_openMC(devid, &rbuf, &wbuf, &backend[devid]);

    //xhib_set_sendfuncMC(devid, SENDFUNC_PIOW);
    //rbuf = piowbuf;

    xhib_set_test_modeMC(devid, TESTMODE_REFDESIGN_RAM);

    printf("clear DMA buf...\n");

    for (i = 0; i < size+10; i++) {
	rbuf[i] = 0x123456789abc0000ll|i;
	wbuf[i] = 0xfedcba9876540000ll|i;
	//      fprintf(stderr, "rbuf[0x%02x]: 0x%016llx\n", i, rbuf[i]);
    }

#define NLOOP (1e8)

    for (size = 32; size <= 4096; size *= 2) {
	for (ntry = 0; ntry < 1; ntry++) {
	    get_cputime(&lt, &st);
	    for (j = 0; j < NLOOP/size; j++) {
		xhib_sendMC(devid, size, rbuf);
	    }
	    get_cputime(&lt, &st);
	    printf("size: %d DMA read: %f sec  %f MB/s\n",
		   size*8, lt, sizeof(UINT64)*NLOOP/1e6/lt);
	}
    }
    xhib_set_test_modeMC(devid, TESTMODE_NONE);
    xhib_closeMC(devid);
}


void
showstatus(int argc, char **argv)
{
    int i;
    int plda_csr, busmode;
    int devid;
    double freq;

    xhib_set_nclusters(NPCIXMEM);

    for (devid = 0; devid < NPCIXMEM; devid++) {
	xhib_openMC(devid, &rbuf, &wbuf, &backend[devid]);

	fprintf(stderr, "\n## xhib%d:\n", devid);
	plda_csr = TBconfigRead(devid, 0x44); /* PLDA Core Status Register at 0x44 */
	switch ((plda_csr>>28)&0x3) {
	  case 0:
	    freq = 33.0;
	    break;
	  case 1:
	    freq = 66.0;
	    break;
	  case 2:
	    freq = 100.0;
	    break;
	  case 3:
	    freq = 133.0;
	    break;
	}
	busmode = (plda_csr>>30)&0x1;
	fprintf(stderr, "PCI bus freq.: %3.0f MHz  Bus mode: %s \n",
		freq, busmode ? "PCI-X" : "PCI");
	fprintf(stderr, "configration register:\n");
	for (i = 0; i < 16; i++) {
	    fprintf(stderr, "0x%08x: 0x%08x\n",
		    i*4, TBconfigRead(devid, i*4));
	}
	if (busmode) {
	    fprintf(stderr, "---- PCI-X extension ----\n");
	    for (; i < 24; i++) {
		fprintf(stderr, "0x%08x: 0x%08x\n",
			i*4, TBconfigRead(devid, i*4));
	    }
	}
	xhib_closeMC(devid);
    }
}

void
configread(int argc, char **argv)
{
    int devid = 0;
    unsigned long int addr;

    if (argc < 3) {
	showusage(argc, argv);
	exit(1);
    }

    if (argc > 3) {
	devid = atoi(argv[3]);
	if (NPCIXMEM < devid+1) {
	    fprintf(stderr,
		    "too large devid(= %d).\n",
		    devid);
	    exit(1);
	}
    }
    xhib_set_nclusters(NPCIXMEM);
    xhib_openMC(devid, &rbuf, &wbuf, &backend[devid]);
    addr = strtoul(argv[2], (char**)NULL, 16);

    fprintf(stderr, "xhib%d config 0x%08lx: 0x%08x\n",
	    devid, addr, TBconfigRead(devid, addr));

    xhib_closeMC(devid);
}

void
configwrite(int argc, char **argv)
{
    int devid = 0;
    unsigned long int addr, val;

    if (argc < 4) {
	showusage(argc, argv);
	exit(1);
    }

    if (argc > 4) {
	devid = atoi(argv[4]);
	if (NPCIXMEM < devid+1) {
	    fprintf(stderr,
		    "too large devid(= %d).\n",
		    devid);
	    exit(1);
	}
    }
    xhib_set_nclusters(NPCIXMEM);
    xhib_openMC(devid, &rbuf, &wbuf, &backend[devid]);
    addr = strtoul(argv[2], (char**)NULL, 16);
    val = strtoul(argv[3], (char**)NULL, 16);
    fprintf(stderr, "write to xhib%d config 0x%08lx value 0x%08lx\n",
	    devid, addr, val);
    TBconfigWrite(devid, addr, val);
    xhib_closeMC(devid);
}

void
regread(int argc, char **argv)
{
    int devid = 0;
    unsigned long int addr;

    if (argc < 3) {
	showusage(argc, argv);
	exit(1);
    }

    if (argc > 3) {
	devid = atoi(argv[3]);
	if (NPCIXMEM < devid+1) {
	    fprintf(stderr,
		    "too large devid(= %d).\n",
		    devid);
	    exit(1);
	}
    }
    xhib_set_nclusters(NPCIXMEM);
    xhib_openMC(devid, &rbuf, &wbuf, &backend[devid]);
    addr = strtoul(argv[2], (char**)NULL, 16);

    fprintf(stderr, "xhib%d 0x%08lx: 0x%08x\n",
	    devid, addr, TBmemRead(devid, addr));

    xhib_closeMC(devid);
}

void
regwrite(int argc, char **argv)
{
    int devid = 0;
    unsigned long int addr, val;

    if (argc < 4) {
	showusage(argc, argv);
	exit(1);
    }

    if (argc > 4) {
	devid = atoi(argv[4]);
	if (NPCIXMEM < devid+1) {
	    fprintf(stderr,
		    "too large devid(= %d).\n",
		    devid);
	    exit(1);
	}
    }
    xhib_set_nclusters(NPCIXMEM);
    xhib_openMC(devid, &rbuf, &wbuf, &backend[devid]);
    addr = strtoul(argv[2], (char**)NULL, 16);
    val = strtoul(argv[3], (char**)NULL, 16);
    fprintf(stderr, "write to xhib%d 0x%08lx value 0x%08lx\n",
	    devid, addr, val);
    TBmemWrite(devid, addr, val);
    xhib_closeMC(devid);
}

void
stopdmaw(int argc, char **argv)
{
    int i, size, nword;
    int devid;

    xhib_set_nclusters(NPCIXMEM);

    fprintf(stderr, "stop DMA channel operation (host <- XHIB)");
    for (devid = 0; devid < NPCIXMEM; devid++) {
	xhib_openMC(devid, &rbuf, &wbuf, &backend[devid]);
	fprintf(stderr, "XHIB[%d]\n", devid);
	TBmemWrite(devid, DMASTAT, 0x80000000);
	xhib_closeMC(devid);
    }
}

void
clearfifo(int argc, char **argv)
{
    int i;
    int devid;
    int datacnt;

    xhib_set_nclusters(NPCIXMEM);

    fprintf(stderr, "clear XHIB-internal FIFO \n");
    for (devid = 0; devid < NPCIXMEM; devid++) {
	xhib_openMC(devid, &rbuf, &wbuf, &backend[devid]);

	fprintf(stderr, "clear XHIB[%d] FIFO...\n", devid);
	while (datacnt = 8 * (TBmemRead(devid, DMAMISC) & 0x7ff)) {
	    fprintf(stderr, "m_dma1_datacnt: %d byte(s)\n", datacnt);
	    usleep(3000000/datacnt);
	    xhib_recvMC(devid, 1, wbuf);
	}

	fprintf(stderr, "...cleared xhib[%d] FIFO\n", devid);

	xhib_closeMC(devid);
    }
}

void
showdmastatus(int argc, char **argv)
{
    int i;
    int devid;

    xhib_set_nclusters(NPCIXMEM);

    fprintf(stderr, "show DMA status \n");
    for (devid = 0; devid < NPCIXMEM; devid++) {
	xhib_openMC(devid, &rbuf, &wbuf, &backend[devid]);

	fprintf(stderr, "\nXHIB[%d]\n\n", devid);

	fprintf(stderr, "PIO write (host->EHIB)\n");
	fprintf(stderr, "    swap_sram: %d\n", (TBmemRead(devid, DMAMISC)>>27)&1);
	fprintf(stderr, "    sram0_wlock: %d\n", (TBmemRead(devid, DMAMISC)>>25)&1);
	fprintf(stderr, "    sram1_wlock: %d\n", (TBmemRead(devid, DMAMISC)>>26)&1);
	fprintf(stderr, "    sram_wcnt: %d\n", (TBmemRead(devid, DMAMISC)>>12)&0x1ff);
	fprintf(stderr, "\n");

	fprintf(stderr, "DMA0 (host->XHIB)\n");
	fprintf(stderr, "    dma0_done: %d\n", (TBmemRead(devid, DMAMISC) >> 28)&1);
	fprintf(stderr, "    data to be transferred:    %d byte\n", TBmemRead(devid, DMA0SIZE));
	fprintf(stderr, "    command & status register: 0x%08x\n", TBmemRead(devid, DMA0CMD));
	fprintf(stderr, "\n");

	fprintf(stderr, "DMA1 (host<-XHIB)\n");
	fprintf(stderr, "    dma1_done: %d\n", (TBmemRead(devid, DMAMISC) >> 29)&1);
	fprintf(stderr, "    m_dma1_datacnt:            %d byte (%d particles)\n",
		8*(TBmemRead(devid, DMAMISC) & 0x7ff), (TBmemRead(devid, DMAMISC) & 0x7ff)/3);
	fprintf(stderr, "    data to be transferred:    %d byte\n", TBmemRead(devid, DMA1SIZE));
	fprintf(stderr, "    command & status register: 0x%08x\n", TBmemRead(devid, DMA1CMD));
	fprintf(stderr, "\n");

	xhib_closeMC(devid);
    }
}

void
pioread(int argc, char **argv)
{
    int devid = 0;
    unsigned long int baddr, waddr;

    if (argc < 3) {
	showusage(argc, argv);
	exit(1);
    }

    if (argc > 3) {
	devid = atoi(argv[3]);
	if (NPCIXMEM < devid+1) {
	    fprintf(stderr,
		    "too large devid(= %d).\n",
		    devid);
	    exit(1);
	}
    }
    xhib_set_nclusters(NPCIXMEM);
    xhib_openMC(devid, &rbuf, &wbuf, &backend[devid]);
    baddr = strtoull(argv[2], (char**)NULL, 16);

    waddr = baddr >> 3;
    baddr = waddr << 3;
    fprintf(stderr, "backend%d 0x%08lx: 0x%016llx\n",
	    devid, baddr, backend[devid][waddr]);

    xhib_closeMC(devid);
}

void
piowrite(int argc, char **argv)
{
    int devid = 0;
    unsigned long int baddr, waddr;
    UINT64 val;

    if (argc < 4) {
	showusage(argc, argv);
	exit(1);
    }

    if (argc > 4) {
	devid = atoi(argv[4]);
	if (NPCIXMEM < devid+1) {
	    fprintf(stderr,
		    "too large devid(= %d).\n",
		    devid);
	    exit(1);
	}
    }
    xhib_set_nclusters(NPCIXMEM);
    xhib_openMC(devid, &rbuf, &wbuf, &backend[devid]);
    baddr = strtoul(argv[2], (char**)NULL, 16);
    val = strtoull(argv[3], (char**)NULL, 16);

    waddr = baddr >> 3;
    baddr = waddr << 3;
    fprintf(stderr, "write to backend%d 0x%08lx value 0x%016llx\n",
	    devid, baddr, val);
    backend[devid][waddr] = val;
    xhib_closeMC(devid);
}

#define MEGA (1e6)
void
rawperf(int argc, char **argv)
{
    int devid = 0;
    int i, j, ntry, nng, size0, off;
    int size; /* in 64-bit words */
    double sized, ratio, nloop;
    double lt = 0.0, st = 0.0;
    UINT64 *b;

    if (argc < 2) {
	showusage(argc, argv);
	exit(1);
    }

    xhib_set_nclusters(NPCIXMEM);
    xhib_openMC(devid, &rbuf, &wbuf, &backend[devid]);
    xhib_set_test_modeMC(devid, TESTMODE_REFDESIGN_RAM);
    b = backend[devid];
    ratio = 1.1;

#if 0
    // DMA read
    nloop = 1e7;
    printf("\n#\n# DMA read\n#\n");
    for (sized = size = 2; size < 1024+1; sized *= ratio, size = sized) {
	get_cputime(&lt, &st);
	for (j = 0; j < nloop/size; j++) {
	    xhib_sendMC(devid, size, rbuf);
	}
	get_cputime(&lt, &st);
	printf("%ld byte    %f sec    %f MB/s\n",
               size*sizeof(UINT64), lt, nloop*sizeof(UINT64)/MEGA/lt);
	fflush(stdout);
    }

    // DMA write
    nloop = 1e7;
    printf("\n#\n# DMA write\n#\n");
    for (sized = size = 2; size < 1024+1; sized *= ratio, size = sized) {
	get_cputime(&lt, &st);
	for (j = 0; j < nloop/size; j++) {
	    xhib_recvMC(devid, size, wbuf);
	}
	get_cputime(&lt, &st);
	printf("%ld byte    %f sec    %f MB/s\n",
               size*sizeof(UINT64), lt, nloop*sizeof(UINT64)/MEGA/lt);
	fflush(stdout);
    }
#endif

    // PIO write
    nloop = 1e7;

    rbuf = piowbuf;
    printf("\n#\n# PIO write\n#\n");
    for (sized = size = 200; size < 1024+1; sized *= ratio, size = sized) {
	get_cputime(&lt, &st);
	for (j = 0; j < nloop/size; j++) {
	    for (i = 0; i < size; i++) {
	        b[i] = piowbuf[i];
	    }
//	    _mm_mfence();
	}
	get_cputime(&lt, &st);
	printf("%ld byte    %f sec    %f MB/s\n",
               size*sizeof(UINT64), lt, nloop*sizeof(UINT64)/MEGA/lt);
	fflush(stdout);
    }

#if 0
    // PIO read
    nloop = 1e6;

    rbuf = piowbuf;
    printf("\n#\n# PIO read\n#\n");
    for (sized = size = 2; size < 1024+1; sized *= ratio, size = sized) {
	get_cputime(&lt, &st);
	for (j = 0; j < nloop/size; j++) {
	    for (i = 0; i < size; i++) {
	        piowbuf[i] = b[i];
	    }
//	    _mm_mfence();
	}
	get_cputime(&lt, &st);
	printf("%ld byte    %f sec    %f MB/s\n",
               size*sizeof(UINT64), lt, nloop*sizeof(UINT64)/MEGA/lt);
	fflush(stdout);
    }
#endif


    xhib_set_test_modeMC(devid, TESTMODE_NONE);
    xhib_closeMC(devid);
}

void
get_cputime(double *splittime, double *laptime)
{
    struct timeval x;

    gettimeofday(&x, NULL);

    *splittime = x.tv_sec + x.tv_usec/1000000.0 - *laptime;
    *laptime = x.tv_sec + x.tv_usec/1000000.0;
}
