/*
 * PCI-X Host Interface Bridge (XHIB) device driver
 * for GRAPE-5X and GRAPE-6AX
 * 
 * Modified by KFCR. should work with kernel version 2.6.26 (2009/06/06)
 * 
 * DMA strategy:
 * + init_module() allocate DMA read&write buf in kernel space
 * + GET_DMAR_PA ioctl returns phys addr of the read buf
 * + GET_DMAW_PA ioctl returns phys addr of the write buf
 * + specify mmap behavior by SET_MAP_MODE ioctl
 * + mmap maps XHIB local registers              if mapmode==MAP_XHIBMEM (BAR0)
 *             XHIB backend device               if mapmode==MAP_BACKEND (BAR1)
 *             XHIB internal double SRAM buffers if mapmode==MAP_DBLBUF  (BAR2)
 *             DMA read buf in kernel space      if mapmode==MAP_DMARBUF (main memory)
 *             DMA write buf in kernel space     if mapmode==MAP_DMAWBUF (main memory)
 */

#ifdef __alpha
#define USE_48_BIT_KSEG
#endif /* __alpha */

#include <asm/current.h>

#if defined(CONFIG_SMP) && !defined(__SMP__)
#define __SMP__
#endif

#include <linux/fs.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/pci.h>
#include <linux/mm.h>
#include <linux/sched.h>
#include <linux/version.h>
#include <asm/io.h>
#include <asm/page.h>
#include <asm/pgtable.h>
#include <asm/uaccess.h>
#include "pcixmem.h"

#if 0
#define Cprintk(fmt, args...) printk(fmt, ## args)
#else
#define Cprintk(fmt, args...)
#endif

#ifndef LINUX_VERSION_CODE
#define LINUX_VERSION_CODE 0x020600
#warning LINUX_VERSION_CODE undefined. assume version 0x020600.
#endif

/*
 * vm-operation method to handle page fault.
 */
#if LINUX_VERSION_CODE < 0x02061a // earlier than 2.6.26. use nopage() method.
#define HAS_VMOP_FAULT (0)
#else // 2.6.26 or later. use fault() method.
#define HAS_VMOP_FAULT (1)
#endif

/*
 * kernel interface to initialize a page reference count.
 */
#if LINUX_VERSION_CODE < 0x020613 // earlier than 2.6.19. use get_page().
#define HAS_INIT_PAGE_COUNT (0)
#else // 2.6.13 or later. use init_page_count().
#define HAS_INIT_PAGE_COUNT (1)
#endif

#if LINUX_VERSION_CODE < 0x020400
#warning "older than kernel 2.4"
#elif LINUX_VERSION_CODE < 0x020600
#warning "older than kernel 2.6 but newer than 2.3"
#else 
// #warning "kernel 2.6 or later"
#endif

/* kernel 2.2 or earlier */
#if LINUX_VERSION_CODE < 0x020400
#define pci_resource_start(dev, region) (dev->base_address[region])
#ifndef VMA_OFFSET
#define VMA_OFFSET(vma) (vma->vm_offset)
#endif
#ifndef VM_RESERVED
#define VM_RESERVED (0)
#endif
#define Page unsigned long
int
pci_enable_device(dev)
{
    return (0);
}
static Page
virt_to_page(unsigned long virtual)
{
    return virtual;
}
static void
get_page(Page pageptr)
{
    atomic_inc(&mem_map[MAP_NR(pageptr)].count);
}

#else /* kernel 2.4 or later */

#ifndef VMA_OFFSET
#define VMA_OFFSET(vma) ((vma->vm_pgoff)<<PAGE_SHIFT)
#endif
#define Page struct page*
#endif /* LINUX_VERSION_CODE */

/* file operation methods */
static int pcixmem_open(struct inode *ip, struct file *fp);
static int pcixmem_close(struct inode *ip, struct file *fp);
static int n_pcixmem_exist(int vendorid, int deviceid);
static int pcixmem_ioctl(struct inode *ip, struct file *fp,
			unsigned int cmd, unsigned long arg);
static int pcixmem_mmap(struct file *fp, struct vm_area_struct *vp);

/* vm operation methods */
#if HAS_VMOP_FAULT
static int pcixmem_dmar_vma_fault(struct vm_area_struct *vma, struct vm_fault *vmf);
static int pcixmem_dmaw_vma_fault(struct vm_area_struct *vma, struct vm_fault *vmf);
#else
static Page pcixmem_dmar_vma_nopage(struct vm_area_struct *vp, unsigned long address, int *type);
static Page pcixmem_dmaw_vma_nopage(struct vm_area_struct *vp, unsigned long address, int *type);
#endif
				    
static void pcixmem_vma_open(struct vm_area_struct *vp);
static void pcixmem_vma_close(struct vm_area_struct *vp);

static void release_pages(int order, struct pci_dev *dev, unsigned long *va, dma_addr_t pa);

/* local defs */
static int major;
static struct pci_dev *pcixmem[NPCIXMEM];
static unsigned long *dmarbuf[NPCIXMEM];
static unsigned long *dmawbuf[NPCIXMEM];
static dma_addr_t dmarbufpa[NPCIXMEM];
static dma_addr_t dmawbufpa[NPCIXMEM];
static unsigned int mapmode[NPCIXMEM];
static int npcixmem; /* # of pcixmem devices found */
static struct file_operations fops =
{
    .ioctl = pcixmem_ioctl,
    .mmap = pcixmem_mmap,
    .open = pcixmem_open,
    .release = pcixmem_close,
};
static struct vm_operations_struct dmar_vmops =
{
    .open = pcixmem_vma_open,
    .close = pcixmem_vma_close,
#if HAS_VMOP_FAULT
    .fault = pcixmem_dmar_vma_fault,
#else
    .nopage = pcixmem_dmar_vma_nopage,
#endif
};

static struct vm_operations_struct dmaw_vmops =
{
    .open = pcixmem_vma_open,
    .close = pcixmem_vma_close,
#if HAS_VMOP_FAULT
    .fault = pcixmem_dmaw_vma_fault,
#else
    .nopage = pcixmem_dmaw_vma_nopage,
#endif
};

static void
release_pages(int order, struct pci_dev *dev,
	   unsigned long *va, dma_addr_t pa)
{
  int i;
  Page toppg = virt_to_page((unsigned long)va);

  for (i = 1; i < (1 << order); i++) { /* note that i starts from 1, not from 0.
					  reference count of the 1st page is automatically
					  handled by pci_free/alloc_consistent. */
      put_page_testzero(toppg + i);
  }
  pci_free_consistent(dev, PAGE_SIZE * (1<<order), va, pa);
}

int
pcixmem_init(void)
{
    int i, ii;
    Page pageptr;

    /* register this device driver */
    major = register_chrdev(0, DEVNAME, &fops);

    /* init static vars */
    for (i = 0; i < NPCIXMEM; i++) {
	pcixmem[i] = NULL;
	dmarbuf[i] = NULL;
	dmawbuf[i] = NULL;
	mapmode[i] = MAP_XHIBMEM;
    }

    /* look for pcixmem device(s) */
    npcixmem = n_pcixmem_exist(0x1556, 0x5555);
    if (!npcixmem) {
	return (-ENODEV);
    }
    Cprintk(KERN_ALERT "bus: %d func: %d bar2 %08lx\n",
	   pcixmem[0]->bus->number,
	   pcixmem[0]->devfn,
	   pcixmem[0]->base_address[2]);

    /* DMA configuration */
    for (i = 0; i < npcixmem; i++) {
	/* set DMA masks */
	if (pci_set_dma_mask(pcixmem[i], 0xffffffff)) {
	    printk(KERN_WARNING "pcixmem: No suitable DMA available.\n");
	    return (-ENODEV);
	}
	printk(KERN_WARNING "pcixmem: DMA mask set to 32-bit.\n");

	if (pci_set_consistent_dma_mask(pcixmem[i], 0xffffffff)) {
	    printk(KERN_WARNING "pcixmem: No suitable consistent DMA available.\n");
	    return (-ENODEV);
	}
	printk(KERN_WARNING "pcixmem: consistent DMA mask set to 32-bit.\n");

	/* allocate DMA read buffers */
	dmarbuf[i] = pci_alloc_consistent(pcixmem[i], XHIBRAMSIZE, &dmarbufpa[i]);
	if (NULL == dmarbuf[i]) {
	    printk(KERN_ALERT "pcixmem: pci_alloc_consistent failed\n");
	    for (ii = i-1; ii >= 0; ii--) {
	        release_pages(XHIBRAMORDER, pcixmem[ii], dmarbuf[ii], dmarbufpa[ii]);
	    }
	    for (ii = i-1; ii >= 0; ii--) {
	        release_pages(XHIBRAMORDER, pcixmem[ii], dmawbuf[ii], dmawbufpa[ii]);
	    }
	    return (-ENOMEM);
	}
	pageptr = virt_to_page((unsigned long)dmarbuf[i]);
	for (ii = 1; ii < (1 << XHIBRAMORDER); ii++) {
#if HAS_INIT_PAGE_COUNT
            init_page_count(pageptr + ii);
#else
            get_page(pageptr + ii);
#endif
	}

	/* allocate DMA write buffers */
	dmawbuf[i] = pci_alloc_consistent(pcixmem[i], XHIBRAMSIZE, &dmawbufpa[i]);
	if (NULL == dmawbuf[i]) {
	    printk(KERN_ALERT "pcixmem: pci_alloc_consistent failed\n");
	    for (ii = i; ii >= 0; ii--) {
	        release_pages(XHIBRAMORDER, pcixmem[ii], dmarbuf[ii], dmarbufpa[ii]);
	    }
	    for (ii = i-1; ii >= 0; ii--) {
	        release_pages(XHIBRAMORDER, pcixmem[ii], dmawbuf[ii], dmawbufpa[ii]);
	    }
	    return (-ENOMEM);
	}
	pageptr = virt_to_page((unsigned long)dmawbuf[i]);
	for (ii = 1; ii < (1 << XHIBRAMORDER); ii++) {
#if HAS_INIT_PAGE_COUNT
            init_page_count(pageptr + ii);
#else
            get_page(pageptr + ii);
#endif
	}
    }

    // just for test
    for (i = 0; i < 4097; i++) {
      dmawbuf[0][i] = 0x12345678abc00000ll | i;
    }

    printk(KERN_ALERT "pcixmem installed\n");

    return (0);
}

void
pcixmem_exit(void)
{
    int i;

    for (i = 0; i < npcixmem; i++) {
        release_pages(XHIBRAMORDER, pcixmem[i], dmarbuf[i], dmarbufpa[i]);
        release_pages(XHIBRAMORDER, pcixmem[i], dmawbuf[i], dmawbufpa[i]);
    }

    unregister_chrdev(major, DEVNAME);
    printk(KERN_ALERT "pcixmem uninstalled\n");
}

module_init(pcixmem_init);
module_exit(pcixmem_exit);
MODULE_LICENSE("Dual BSD/GPL");

static int
pcixmem_open(struct inode *ip, struct file *fp)
{
    Cprintk(KERN_ALERT "pcixmem_open\n");
    /*
    MOD_INC_USE_COUNT;
    */
    return (0);
}

static int
pcixmem_close(struct inode *ip, struct file *fp)
{
    Cprintk(KERN_ALERT "pcixmem_close\n");
    /*
    MOD_DEC_USE_COUNT;
    */
    return (0);
}

static int
pcixmem_ioctl(struct inode *ip, struct file *fp, 
	     unsigned int cmd, unsigned long arg)
{
    int minor = MINOR(ip->i_rdev);
    Cprintk(KERN_ALERT "pcixmem_ioctl cmd: %x\n", (int) cmd);

    /* exec cmd with argument arg */
    switch (cmd) {
      case WRITE_CFG: {
          struct long_access myarg;
	  if (copy_from_user(&myarg, (struct long_access *)arg, sizeof(myarg))) return (-EFAULT);
	  pci_write_config_dword(pcixmem[minor], myarg.addr, myarg.data);

	  Cprintk(KERN_ALERT "pcixmem_ioctl done WRITE_CFG\n");

	  break;
      }
      case READ_CFG: {
	  struct long_access myarg;

	  Cprintk(KERN_ALERT "pcixmem_ioctl will READ_CFG\n");
	  if (copy_from_user(&myarg, (struct long_access *)arg, sizeof(myarg))) return (-EFAULT);

	  pci_read_config_dword(pcixmem[minor], myarg.addr, &(myarg.data));
	  Cprintk(KERN_ALERT "pcixmem_ioctl READ_CFG  addr: %x  data: %x\n", myarg.addr, myarg.data);

	  if (copy_to_user((struct long_access *)arg, &myarg, sizeof(myarg))) return (-EFAULT);

	  Cprintk(KERN_ALERT "pcixmem_ioctl done READ_CFG\n");

	  break;

      }
      case GET_DMAR_PA: {
	  unsigned long baddr;

	  Cprintk(KERN_ALERT "pcixmem_ioctl will GET_DMAR_PA\n");

	  baddr = dmarbufpa[minor];
	  if (copy_to_user((unsigned int *)arg, &baddr, sizeof(baddr))) return (-EFAULT);

	  Cprintk(KERN_ALERT "GET_DMAR_PA: DMA read buf bus addr: %016lx\n", baddr);
	  Cprintk(KERN_ALERT "pcixmem_ioctl done GET_DMAR_PA\n");
	  break;
      }
      case GET_DMAW_PA: {
	  unsigned long baddr;

	  Cprintk(KERN_ALERT "pcixmem_ioctl will GET_DMAW_PA\n");

	  baddr = dmawbufpa[minor];
	  if (copy_to_user((unsigned int *)arg, &baddr, sizeof(baddr))) return (-EFAULT);

	  Cprintk(KERN_ALERT "GET_DMAW_PA: DMA write buf bus addr: %016lx\n", baddr);
	  Cprintk(KERN_ALERT "pcixmem_ioctl done GET_DMAW_PA\n");
	  break;
      }
      case SET_MAP_MODE: {
          Cprintk(KERN_ALERT "pcixmem_ioctl will SET_MAP_MODE\n");

	  mapmode[minor] = (unsigned int)arg;
	  Cprintk(KERN_ALERT "SET_MAP_MODE: mapmode: %d\n", mapmode[minor]);

	  Cprintk(KERN_ALERT "pcixmem_ioctl done SET_MAP_MODE\n");

	  break;
      }
      default:
	return (-EINVAL);
    }
    return (0);
}

#if HAS_VMOP_FAULT // implement fault() method.

static int
pcixmem_dmar_vma_fault(struct vm_area_struct *vma, struct vm_fault *vmf)
{
    int minor = MINOR(vma->vm_file->f_dentry->d_inode->i_rdev);
    Page pageptr;
    unsigned long offset = vmf->pgoff << PAGE_SHIFT;
    // address - vma->vm_start + VMA_OFFSET(vma);

    Cprintk(KERN_ALERT "pcixmem_dmar_vma_fault minor: %d\n", minor);
    pageptr = virt_to_page((unsigned long)dmarbuf[minor] + offset);

    vmf->page = pageptr;
    get_page(pageptr);

    return 0;
}

static int
pcixmem_dmaw_vma_fault(struct vm_area_struct *vma, struct vm_fault *vmf)
{
    int minor = MINOR(vma->vm_file->f_dentry->d_inode->i_rdev);
    Page pageptr;
    unsigned long offset = vmf->pgoff << PAGE_SHIFT;
    // address - vma->vm_start + VMA_OFFSET(vma);

    Cprintk(KERN_ALERT "pcixmem_dmaw_vma_fault minor: %d\n", minor);
    pageptr = virt_to_page((unsigned long)dmawbuf[minor] + offset);

    vmf->page = pageptr;
    get_page(pageptr);

    return 0;
}

#else // implement nopage() method.

static Page
pcixmem_dmar_vma_nopage(struct vm_area_struct *vp, unsigned long address, int *type)
{
    int minor = MINOR(vp->vm_file->f_dentry->d_inode->i_rdev);
    Page pageptr;
    unsigned long offset = address - vp->vm_start + VMA_OFFSET(vp);

    Cprintk(KERN_ALERT "pcixmem_dmar_vma_nopage minor: %d\n", minor);
    pageptr = virt_to_page((unsigned long)dmarbuf[minor] + offset);
    get_page(pageptr);

    return pageptr;
}

static Page
pcixmem_dmaw_vma_nopage(struct vm_area_struct *vp, unsigned long address, int *type)
{
    int minor = MINOR(vp->vm_file->f_dentry->d_inode->i_rdev);
    Page pageptr;
    unsigned long offset = address - vp->vm_start + VMA_OFFSET(vp);

    Cprintk(KERN_ALERT "pcixmem_dmaw_vma_nopage minor: %d\n", minor);
    pageptr = virt_to_page((unsigned long)dmawbuf[minor] + offset);
    get_page(pageptr);

    Cprintk(KERN_ALERT "pageptr: 0x%08x  dmawbuf[%d]: 0x%08x  offset: 0x%08x\n",
	   pageptr, minor, dmawbuf[minor], offset);

    return pageptr;
}

#endif

static void
pcixmem_vma_open(struct vm_area_struct *vp)
{
    Cprintk(KERN_ALERT "pcixmem_vma_open\n");
    // MOD_INC_USE_COUNT;
}

static void
pcixmem_vma_close(struct vm_area_struct *vp)
{
    Cprintk(KERN_ALERT "pcixmem_vma_close\n");
    // MOD_DEC_USE_COUNT;
}

static int
pcixmem_mmap(struct file *fp, struct vm_area_struct *vp)
{
    unsigned long off = VMA_OFFSET(vp);
    unsigned long virtual = vp->vm_start;
    unsigned long physical;
    unsigned long vsize = vp->vm_end - vp->vm_start;
    int ret = 0;
    int minor = MINOR(fp->f_dentry->d_inode->i_rdev);
    int bar;

    switch (mapmode[minor]) {
      case MAP_XHIBMEM: /* map XHIB local memory */
	bar = 0;
	break;
      case MAP_BACKEND: /* map XHIB backend address space */
	bar = 1;
	break;
      case MAP_DBLBUF: /* map XHIB internal double SRAM buffers */
	bar = 2;
	break;
      default:
	bar = 0;
    }

    switch (mapmode[minor]) {
      case MAP_XHIBMEM: /* map XHIB local memory */
      case MAP_BACKEND: /* map XHIB backend address space */
      case MAP_DBLBUF: /* map XHIB internal double SRAM buffers */

	physical  = pci_resource_start(pcixmem[minor], bar) & PCI_BASE_ADDRESS_MEM_MASK;
	/* map BAR to the user space */
	/* printk(KERN_ALERT "bar%d physical: 0x%0x\n", bar, physical); */
	physical += off;
	// vp->vm_page_prot = pgprot_noncached(vp->vm_page_prot);
	vp->vm_flags |= VM_RESERVED;

#if LINUX_VERSION_CODE < 0x020503 /* earlier than 2.5.3 */
	ret = remap_page_range(virtual, physical, vsize, vp->vm_page_prot);
#elif LINUX_VERSION_CODE < 0x020610 /* earlier than 2.6.10 */
     ret = remap_pfn_range(vp, virtual, physical >> PAGE_SHIFT, vsize, vp->vm_page_prot);
     //      ret = remap_page_range(vp, virtual, physical, vsize, vp->vm_page_prot);
#else /* 2.6.10 or later */
      ret = remap_pfn_range(vp, virtual, physical >> PAGE_SHIFT, vsize, vp->vm_page_prot);
#endif

	if (ret) return (-EAGAIN);
	/* printk(KERN_ALERT "MAP_XHIBMEM done\n"); */
	break;

    case MAP_DMARBUF: /* map DMA read buf */
	vp->vm_ops = &dmar_vmops;
	vp->vm_page_prot = pgprot_noncached(vp->vm_page_prot);
	vp->vm_flags |= VM_RESERVED | VM_IO;
	pcixmem_vma_open(vp);
	Cprintk(KERN_ALERT "MAP_DMARBUF done\n");
	break;

    case MAP_DMAWBUF: /* map DMA write buf */
	vp->vm_ops = &dmaw_vmops;
	//	vp->vm_page_prot = pgprot_noncached(vp->vm_page_prot);
	vp->vm_flags |= VM_RESERVED | VM_IO;
	pcixmem_vma_open(vp);
	Cprintk(KERN_ALERT "MAP_DMAWBUF done\n");
	break;

    default:
	printk(KERN_ALERT "unknown map mode %d\n", mapmode[minor]);
	ret = 1;
	break;
    }
    if (ret) {
	return (-EAGAIN);
    }
    return (ret);
}

static int
n_pcixmem_exist(int vendorid, int deviceid)
{
    int i;
    struct pci_dev *pcixmem0 = NULL;

    for (i = 0; ; i++) {
	pcixmem[i] = pci_find_device(vendorid, deviceid, pcixmem0);
	pcixmem0 = pcixmem[i];
	if (!pcixmem[i]) {
	    break;
	}
	else if (pci_enable_device(pcixmem[i])) {
	    break;
	}
    }
    if (i == 0) {
	printk(KERN_ALERT "no pcixmem found\n");
	return (-ENODEV);
    }
    else {
	printk(KERN_ALERT "%d pcixmem(s) found\n", i);
	return (i);
    }
}
