/*
 * PCI Host Interface Board device driver
 * for GRAPE-4/5/6TB and PROGRAPE-1
 * 
 * Modified by KFCR. should work with kernel version 2.6.26 (2009/06/06)
 * 
 * DMA strategy:
 * + init_module() allocate DMA buf in kernel space
 * + GET_DMA_INFO ioctl returns phys addr of the buf
 * + specify mmap behavior by SET_MAP_MODE ioctl
 * + mmap maps PHIB local reg if mapmode==MAP_PHIBREG
 *             DMA buf        if mapmode==MAP_DMABUF
 */

#ifdef __alpha
#define USE_48_BIT_KSEG
#endif /* __alpha */

#include <asm/current.h>

#if defined(CONFIG_SMP) && !defined(__SMP__)
#define __SMP__
#endif

#include <linux/fs.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/pci.h>
#include <linux/mm.h>
#include <linux/sched.h>
#include <linux/version.h>
#include <asm/io.h>
#include <asm/page.h>
#include <asm/pgtable.h>
#include <asm/uaccess.h>
#include "pcimem.h"

#if 0
#define Cprintk(fmt, args...) printk(fmt, ## args)
#else
#define Cprintk(fmt, args...)
#endif

#define PAGE_REQ_ORDER (0) /* request 2^order pages for DMA buffer (max 2^5) */
#define DMABUF_BYTES (PAGE_SIZE * (1<<PAGE_REQ_ORDER))

#ifndef LINUX_VERSION_CODE
#define LINUX_VERSION_CODE 0x020600
#warning LINUX_VERSION_CODE undefined. assume version 0x020600.
#endif

/*
 * vm-operation method to handle page fault.
 */
#if LINUX_VERSION_CODE < 0x02061a // earlier than 2.6.26. use nopage() method.
#define HAS_VMOP_FAULT (0)
#else // 2.6.26 or later. use fault() method.
#define HAS_VMOP_FAULT (1)
#endif

/*
 * kernel interface to initialize a page reference count.
 */
#if LINUX_VERSION_CODE < 0x020613 // earlier than 2.6.19. use get_page().
#define HAS_INIT_PAGE_COUNT (0)
#else // 2.6.13 or later. use init_page_count().
#define HAS_INIT_PAGE_COUNT (1)
#endif

#if LINUX_VERSION_CODE < 0x020400
#warning "older than kernel 2.4"
#elif LINUX_VERSION_CODE < 0x020600
#warning "older than kernel 2.6 but newer than 2.3"
#else 
// #warning "kernel 2.6 or later"
#endif

/* kernel 2.2 or earlier */
#if LINUX_VERSION_CODE < 0x020400
#define pci_resource_start(dev, region) (dev->base_address[region])
#ifndef VMA_OFFSET
#define VMA_OFFSET(vma) (vma->vm_offset)
#endif
#ifndef VM_RESERVED
#define VM_RESERVED (0)
#endif
#define Page unsigned long
int
pci_enable_device(dev)
{
    return (0);
}
static Page
virt_to_page(unsigned long virtual)
{
    return virtual;
}
static void
get_page(Page pageptr)
{
    atomic_inc(&mem_map[MAP_NR(pageptr)].count);
}

#else /* kernel 2.4 or later */

#ifndef VMA_OFFSET
#define VMA_OFFSET(vma) ((vma->vm_pgoff)<<PAGE_SHIFT)
#endif
#define Page struct page*
#endif /* LINUX_VERSION_CODE */

/* file operation methods */
static int pcimem_open(struct inode *ip, struct file *fp);
static int pcimem_close(struct inode *ip, struct file *fp);
static int n_pcimem_exist(int vendorid, int deviceid);
static int pcimem_ioctl(struct inode *ip, struct file *fp,
			unsigned int cmd, unsigned long arg);
static int pcimem_mmap(struct file *fp, struct vm_area_struct *vp);

/* vm operation methods */
#if HAS_VMOP_FAULT
static int pcimem_vma_fault(struct vm_area_struct *vma, struct vm_fault *vmf);
#else
static Page pcimem_vma_nopage(struct vm_area_struct *vp, unsigned long address, int *type);
#endif

static void pcimem_vma_open(struct vm_area_struct *vp);
static void pcimem_vma_close(struct vm_area_struct *vp);

static void release_pages(int order, struct pci_dev *dev, unsigned long *va, dma_addr_t pa);

/* local defs */
static int major;
static struct pci_dev *pcimem[NPCIMEM];
static unsigned long *dmabuf[NPCIMEM];
static dma_addr_t dmabufpa[NPCIMEM];
static unsigned int mapmode[NPCIMEM];
static int npcimem; /* # of pcimem devices found */

static struct file_operations fops =
{
    .ioctl = pcimem_ioctl,
    .mmap = pcimem_mmap,
    .open = pcimem_open,
    .release = pcimem_close,
};
static struct vm_operations_struct vmops =
{
    .open = pcimem_vma_open,
    .close = pcimem_vma_close,
#if HAS_VMOP_FAULT
    .fault = pcimem_vma_fault,
#else
    .nopage = pcimem_vma_nopage,
#endif
};

static void
release_pages(int order, struct pci_dev *dev,
	   unsigned long *va, dma_addr_t pa)
{
  int i;
  Page toppg = virt_to_page((unsigned long)va);

  for (i = 1; i < (1 << order); i++) { /* note that i starts from 1, not from 0.
					  reference count of the 1st page is automatically
					  handled by pci_free/alloc_consistent. */
      put_page_testzero(toppg + i);
  }
  pci_free_consistent(dev, PAGE_SIZE * (1<<order), va, pa);
}

int
pcimem_init(void)
{
    int i, ii;
    Page pageptr;

    /* register this device driver */
    major = register_chrdev(0, DEVNAME, &fops);

    /* init static vars */
    for (i = 0; i < NPCIMEM; i++) {
	pcimem[i] = NULL;
	dmabuf[i] = NULL;
	mapmode[i] = MAP_PHIBREG;
    }

    /* look for pcimem device(s) */
    npcimem = n_pcimem_exist(PCI_VENDOR_ID_PLX, PCI_DEVICE_ID_PLX_9080);
    if (!npcimem) {
	return (-ENODEV);
    }

    //    Cprintk(KERN_ALERT "bus: %d func: %d bar2 %08lx\n",
    //	   pcimem[0]->bus->number,
    //	   pcimem[0]->devfn,
    //	   pcimem[0]->base_address[2]);

    for (i = 0; i < npcimem; i++) {
	/* set DMA masks */
	if (pci_set_dma_mask(pcimem[i], 0xffffffff)) {
	    printk(KERN_WARNING "pcimem: No suitable DMA available.\n");
            unregister_chrdev(major, DEVNAME); 
	    return (-ENODEV);
	}
	// printk(KERN_WARNING "pcimem: DMA mask set to 32-bit.\n");

	if (pci_set_consistent_dma_mask(pcimem[i], 0xffffffff)) {
	    printk(KERN_WARNING "pcimem: No suitable consistent DMA available.\n");
            unregister_chrdev(major, DEVNAME); 
	    return (-ENODEV);
	}
	// printk(KERN_WARNING "pcimem: consistent DMA mask set to 32-bit.\n");

	/* allocate DMA read buffers */
	dmabuf[i] = pci_alloc_consistent(pcimem[i], DMABUF_BYTES, &dmabufpa[i]);
	if (NULL == dmabuf[i]) {
	    printk(KERN_ALERT "pcimem: pci_alloc_consistent failed\n");
	    for (ii = i-1; ii >= 0; ii--) {
	        release_pages(PAGE_REQ_ORDER, pcimem[ii], dmabuf[ii], dmabufpa[ii]);
	    }
            unregister_chrdev(major, DEVNAME); 
	    return (-ENOMEM);
	}
	pageptr = virt_to_page((unsigned long)dmabuf[i]);
	for (ii = 1; ii < (1 << PAGE_REQ_ORDER); ii++) {
#if HAS_INIT_PAGE_COUNT
            init_page_count(pageptr + ii);
#else
            get_page(pageptr + ii);
#endif
	}
    }

    printk(KERN_ALERT "pcimem installed\n");
    return (0);
}

void
pcimem_exit(void)
{
    int i;

    for (i = 0; i < npcimem; i++) {
        release_pages(PAGE_REQ_ORDER, pcimem[i], dmabuf[i], dmabufpa[i]);
    }

    unregister_chrdev(major, DEVNAME);
    printk(KERN_ALERT "pcimem uninstalled\n");
}

module_init(pcimem_init);
module_exit(pcimem_exit);
MODULE_LICENSE("Dual BSD/GPL");

static int
pcimem_open(struct inode *ip, struct file *fp)
{
    Cprintk(KERN_ALERT "pcimem_open\n");
    /*
    MOD_INC_USE_COUNT;
    */
    return (0);
}

static int
pcimem_close(struct inode *ip, struct file *fp)
{
    Cprintk(KERN_ALERT "pcimem_close\n");
    /*
    MOD_DEC_USE_COUNT;
    */
    return (0);
}

static int
pcimem_ioctl(struct inode *ip, struct file *fp, 
	     unsigned int cmd, unsigned long arg)
{
    int minor = MINOR(ip->i_rdev);
    Cprintk(KERN_ALERT "pcimem_ioctl cmd: %x\n", (int) cmd);

    Cprintk(KERN_ALERT "pcimem_ioctl minor: %d\n", minor);

    /* exec cmd with argument arg */
    switch (cmd) {
      case WRITE_CFG:
          {
              struct long_access myarg;

              Cprintk(KERN_ALERT "pcimem_ioctl will WRITE_CFG\n");

              if (copy_from_user(&myarg, (struct long_access *)arg, sizeof(myarg))) return (-EFAULT);
              pci_write_config_dword(pcimem[minor], myarg.addr, myarg.data);

              Cprintk(KERN_ALERT "pcimem_ioctl done WRITE_CFG\n");

              break;
          }
    case READ_CFG:
        {
            struct long_access myarg;

            Cprintk(KERN_ALERT "pcimem_ioctl will READ_CFG\n");
            if (copy_from_user(&myarg, (struct long_access *)arg, sizeof(myarg))) return (-EFAULT);

            pci_read_config_dword(pcimem[minor], myarg.addr, &(myarg.data));
            Cprintk(KERN_ALERT "pcimem_ioctl READ_CFG  addr: %x  data: %x\n", myarg.addr, myarg.data);

            if (copy_to_user((struct long_access *)arg, &myarg, sizeof(myarg))) return (-EFAULT);

            Cprintk(KERN_ALERT "pcimem_ioctl done READ_CFG\n");

            break;
        }
      case MAP_BAR2_DENSE_ADDR:
          {
              Cprintk(KERN_ALERT "MAP_BAR2_DENSE_ADDR: do nothing. exist only for compatibility with driver on Tru64\n");
              break;
          }
      case DMA_MAP_LOAD:
          {
              Cprintk(KERN_ALERT "DMA_MAP_LOAD: do nothing. exist only for compatibility with driver on Tru64\n");
              Cprintk(KERN_ALERT "dmabuf[0][0]: 0x%08x\n", dmabuf[0][0]);
              break;
          }
      case GET_DMA_INFO:
          {
              unsigned long baddr;

              Cprintk(KERN_ALERT "pcimem_ioctl will GET_DMA_INFO\n");

              baddr = dmabufpa[minor];
              if (copy_to_user((unsigned int *)arg, &baddr, sizeof(baddr))) return (-EFAULT);

              Cprintk(KERN_ALERT "GET_DMA_INFO: DMA buf bus addr: %016lx\n", baddr);
              Cprintk(KERN_ALERT "pcimem_ioctl done GET_DMA_INFO\n");

              break;
          }
      case SET_MAP_MODE:
          {
              Cprintk(KERN_ALERT "pcimem_ioctl will SET_MAP_MODE\n");

              mapmode[minor] = (unsigned int)arg;
              Cprintk(KERN_ALERT "SET_MAP_MODE: mapmode: %d\n", mapmode[minor]);

              Cprintk(KERN_ALERT "pcimem_ioctl done SET_MAP_MODE\n");

              break;
          }
      default:
	return (-EINVAL);
    }
    return (0);
}

#if HAS_VMOP_FAULT // implement fault() method.

static int
pcimem_vma_fault(struct vm_area_struct *vma, struct vm_fault *vmf)
{
    int minor = MINOR(vma->vm_file->f_dentry->d_inode->i_rdev);
    Page pageptr;
    unsigned long offset = vmf->pgoff << PAGE_SHIFT;
    // address - vma->vm_start + VMA_OFFSET(vma);

    Cprintk(KERN_ALERT "pcimem_vma_fault minor: %d\n", minor);
    pageptr = virt_to_page((unsigned long)dmabuf[minor] + offset);

    vmf->page = pageptr;
    get_page(pageptr);

    return 0;
}

#else // implement nopage() method.

static Page
pcimem_vma_nopage(struct vm_area_struct *vp, unsigned long address, int *type)
{
    int minor = MINOR(vp->vm_file->f_dentry->d_inode->i_rdev);
    Page pageptr;
    unsigned long offset = address - vp->vm_start + VMA_OFFSET(vp);

    Cprintk(KERN_ALERT "pcimem_vma_nopage minor: %d\n", minor);
    offset >>= 2; /* convert from byte address to 4-byte address */
    pageptr = virt_to_page(dmabuf[minor] + offset);
    get_page(pageptr);

    return pageptr;
}

#endif

static void
pcimem_vma_open(struct vm_area_struct *vp)
{
    Cprintk(KERN_ALERT "pcimem_vma_open\n");
    //    MOD_INC_USE_COUNT;
}

static void
pcimem_vma_close(struct vm_area_struct *vp)
{
    Cprintk(KERN_ALERT "pcimem_vma_close\n");
    //    MOD_DEC_USE_COUNT;
}

static int
pcimem_mmap(struct file *fp, struct vm_area_struct *vp)
{
    unsigned long off = VMA_OFFSET(vp);
    unsigned long virtual = vp->vm_start;
    unsigned long physical;
    unsigned long vsize = vp->vm_end - vp->vm_start;
    int ret = 0;
    int minor = MINOR(fp->f_dentry->d_inode->i_rdev);

    switch (mapmode[minor]) {
    case MAP_PHIBREG: /* map PHIB local reg */
	physical  = pci_resource_start(pcimem[minor], 2) & PCI_BASE_ADDRESS_MEM_MASK;
	physical += off;
#ifdef __alpha
#define BASE2PHYS(a) __pa(dense_mem(a)+((a)&0xffffffffUL))
	physical = BASE2PHYS(physical);
#endif /* __alpha */
	vp->vm_flags |= VM_RESERVED;

#if LINUX_VERSION_CODE < 0x020503 /* earlier than 2.5.3 */
	ret = remap_page_range(virtual, physical, vsize, vp->vm_page_prot);
#elif LINUX_VERSION_CODE < 0x02060a /* earlier than 2.6.10 */
	ret = remap_page_range(vp, virtual, physical, vsize, vp->vm_page_prot);
#else /* 2.6.10 or later */
        ret = remap_pfn_range(vp, virtual, physical >> PAGE_SHIFT, vsize, vp->vm_page_prot);
#endif

	if (ret) return (-EAGAIN);
	Cprintk(KERN_ALERT "MAP_PHIBREG done\n");
	break;

    case MAP_DMABUF: /* map DMA buf */
	vp->vm_ops = &vmops;
	vp->vm_flags |= VM_RESERVED | VM_IO;
	pcimem_vma_open(vp);
	Cprintk(KERN_ALERT "MAP_DMABUF done\n");
	break;

    default:
	printk(KERN_ALERT "unknown map mode %d\n", mapmode[minor]);
	ret = 1;
	break;
    }
    if (ret) {
	return (-EAGAIN);
    }
    return (ret);
}

static int
n_pcimem_exist(int vendorid, int deviceid)
{
    int i;
    struct pci_dev *pcimem0 = NULL;

    for (i = 0; ; i++) {
	pcimem[i] = pci_find_device(vendorid, deviceid, pcimem0);
	pcimem0 = pcimem[i];
	if (!pcimem[i]) {
	    break;
	}
	else if (pci_enable_device(pcimem[i])) {
	    break;
	}
    }
    if (i == 0) {
	printk(KERN_ALERT "no pcimem found\n");
	return (-ENODEV);
    }
    else {
	printk(KERN_ALERT "%d pcimem(s) found\n", i);
	return (i);
    }
}
