/*
 * PCI Host Interface Board device driver
 * for GRAPE-4/5/6TB and PROGRAPE-1
 * 
 * DMA strategy:
 * + init_module() allocate DMA buf in kernel space
 * + GET_DMA_INFO ioctl returns phys addr of the buf
 * + specify mmap behavior by SET_MAP_MODE ioctl
 * + mmap maps PHIB local reg if mapmode==MAP_PHIBREG
 *             DMA buf        if mapmode==MAP_DMABUF
 */

#include <asm/current.h>
#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/pci.h>
#include <linux/mm.h>
#include <linux/sched.h>
#include <asm/io.h>
#include <asm/page.h>
#include <asm/pgtable.h>
#include "pcimem.h"

#if 0
#define Cprintk(fmt, args...) printk(fmt, ## args)
#else
#define Cprintk(fmt, args...)
#endif

#define PAGE_REQ_ORDER (0) /* request 2^order pages for DMA buffer (max 2^5) */

static int pcimem_open(struct inode *ip, struct file *fp);
static int pcimem_close(struct inode *ip, struct file *fp);
static int n_pcimem_exist(int vendorid, int deviceid);
static int pcimem_ioctl(struct inode *ip, struct file *fp,
			unsigned int cmd, unsigned long arg);
static int pcimem_mmap(struct file *fp, struct vm_area_struct *vp);

static int major;
static struct pci_dev *pcimem[NPCIMEM];
static unsigned int *dmabuf[NPCIMEM];
static unsigned int mapmode[NPCIMEM];
static int npcimem; /* # of pcimem devices found */
struct file_operations fops =
{
    NULL, /* lseek */
    NULL, /* read */
    NULL, /* write */
    NULL, /* readdir */
    NULL, /* poll */
    pcimem_ioctl, /* ioctl */
    pcimem_mmap, /* mmap */
    pcimem_open, /* open */
    NULL, /* flush */
    pcimem_close, /* close */
};

int
init_module(void)
{
    int i, ii;
    unsigned long maptop, mapend;
    unsigned long physical;

    /* register this device driver */
    major = register_chrdev(0, DEVNAME, &fops);

    /* init static vars */
    for (i = 0; i < NPCIMEM; i++) {
	pcimem[i] = NULL;
	dmabuf[i] = NULL;
	mapmode[i] = MAP_PHIBREG;
    }

    /* look for pcimem device(s) */
    npcimem = n_pcimem_exist(PCI_VENDOR_ID_PLX, PCI_DEVICE_ID_PLX_9080);
    if (!npcimem) {
	return (-ENODEV);
    }
    Cprintk(KERN_ALERT "bus: %d func: %d bar2 %08lx\n",
	   pcimem[0]->bus->number,
	   pcimem[0]->devfn,
	   pcimem[0]->base_address[2]);
    for (i = 0; i < npcimem; i++) {
	dmabuf[i] = (unsigned int *)__get_free_pages(GFP_KERNEL, PAGE_REQ_ORDER);
	if (NULL == dmabuf[i]) {
	    for (ii = i-1; ii >= 0; ii--) {
		free_pages((unsigned long)dmabuf[ii], PAGE_REQ_ORDER);
	    }
	    return (-ENOMEM);
	}
	/* this is necessary to make remap_page_range() map kernel space correctly
	   note that the way shown in /usr/src/linux/drivers/ does not work */
	physical = (unsigned long)dmabuf[i];
	physical = virt_to_phys((void *)physical);
	physical += PAGE_OFFSET;
	maptop = MAP_NR(physical);
	mapend = MAP_NR(physical + (PAGE_SIZE << PAGE_REQ_ORDER) - 1);
	Cprintk(KERN_ALERT "maptop: 0x%016lx mapend: 0x%016lx\n", maptop, mapend);
	for (ii = maptop; ii <= mapend; ii++) {
	    set_bit(PG_reserved, &mem_map[ii].flags);
	}
    }
    printk(KERN_ALERT "pcimem installed\n");
    return (0);
}

void
cleanup_module(void)
{
    int i, ii;
    unsigned long maptop, mapend;
    unsigned long physical;

    for (i = 0; i < npcimem; i++) {
	/* undo marking the pages as reserved */
	physical = (unsigned long)dmabuf[i];
	physical = virt_to_phys((void *)physical);
	physical += PAGE_OFFSET;
	maptop = MAP_NR(physical);
	mapend = MAP_NR(physical + (PAGE_SIZE << PAGE_REQ_ORDER) - 1);
	for (ii = maptop; ii <= mapend; ii++) {
	    clear_bit(PG_reserved, &mem_map[ii].flags);
	}
	free_pages((unsigned long)dmabuf[i], PAGE_REQ_ORDER);
    }
    unregister_chrdev(major, DEVNAME);
    printk(KERN_ALERT "pcimem uninstalled\n");
}

static int
pcimem_open(struct inode *ip, struct file *fp)
{
    Cprintk(KERN_ALERT "pcimem_open\n");
    /*
    MOD_INC_USE_COUNT;
    */
    return (0);
}

static int
pcimem_close(struct inode *ip, struct file *fp)
{
    Cprintk(KERN_ALERT "pcimem_close\n");
    /*
    MOD_DEC_USE_COUNT;
    */
    return (0);
}

static int
pcimem_ioctl(struct inode *ip, struct file *fp, 
	     unsigned int cmd, unsigned long arg)
{
    int minor = MINOR(ip->i_rdev);
    Cprintk(KERN_ALERT "pcimem_ioctl cmd: %x\n", (int) cmd);

    /* exec cmd with argument arg */
    switch (cmd)
    {
    case WRITE_CFG:
    {
	struct long_access *myarg = (struct long_access *)arg;
	pci_write_config_dword(pcimem[minor], myarg->addr, myarg->data);
	break;
    }
    case READ_CFG:
    {
	struct long_access *myarg = (struct long_access *)arg;
	pci_read_config_dword(pcimem[minor], myarg->addr, &myarg->data);
	break;
    }
    case MAP_BAR2_DENSE_ADDR:
    {
	Cprintk(KERN_ALERT "MAP_BAR2_DENSE_ADDR: do nothing. exist only for compatibility with driver on Tru64\n");
	break;
    }
    case DMA_MAP_LOAD:
    {
	Cprintk(KERN_ALERT "DMA_MAP_LOAD: do nothing. exist only for compatibility with driver on Tru64\n");
	Cprintk(KERN_ALERT "dmabuf[0][0]: 0x%08x\n", dmabuf[0][0]);
	break;
    }
    case GET_DMA_INFO:
    {
	unsigned long *ba = (unsigned long *)arg;
	*ba = virt_to_bus(dmabuf[minor]);
	Cprintk(KERN_ALERT "GET_DMA_INFO: DMA buf bus addr: %016lx\n", *ba);
	break;
    }
    case SET_MAP_MODE:
    {
	mapmode[minor] = (unsigned int)arg;
	Cprintk(KERN_ALERT "SET_MAP_MODE:: mapmode: %d\n", mapmode[minor]);
	break;
    }
    default:
	return (-EINVAL);
    }
    return (0);
}

static int
pcimem_mmap(struct file *fp, struct vm_area_struct *vp)
{
    unsigned long off = vp->vm_offset;
    unsigned long virtual = vp->vm_start;
    unsigned long physical;
    unsigned long vsize = vp->vm_end - vp->vm_start;
    int ret = 0;
    int minor = MINOR(fp->f_dentry->d_inode->i_rdev);

    if (off & (PAGE_SIZE-1)) { /* offset not aligned */
	return (-ENXIO);
    }
    switch (mapmode[minor]) {
    case MAP_PHIBREG: /* map PHIB local reg */
	physical  = (pcimem[minor]->base_address[2] & PCI_BASE_ADDRESS_MEM_MASK);
	physical += off;
#ifdef __alpha
#define BASE2PHYS(a) __pa(dense_mem(a)+((a)&0xffffffffUL))
	physical = BASE2PHYS(physical);
#endif
	ret = remap_page_range(virtual, physical, vsize, vp->vm_page_prot);
	break;
    case MAP_DMABUF: /* map DMA buf */

	physical = (unsigned long)dmabuf[minor];
	physical += off;
	physical = virt_to_phys((void *)(physical));
	physical += PAGE_OFFSET;
	vsize = (PAGE_SIZE << PAGE_REQ_ORDER);
	ret = remap_page_range(virtual, physical, vsize, vp->vm_page_prot);
	break;
    default:
	printk(KERN_ALERT "unknown map mode %d\n", mapmode[minor]);
	ret = 1;
	break;
    }
    if (ret) {
	return (-EAGAIN);
    }
    return (ret);
}


static int
n_pcimem_exist(int vendorid, int deviceid)
{
    int i;
    struct pci_dev *pcimem0 = NULL;

    if (pci_present()) {
	for (i = 0; ; i++) {
	    pcimem[i] = pci_find_device(vendorid, deviceid, pcimem0);
	    pcimem0 = pcimem[i];
	    if (!pcimem[i]) {
		break;
	    }
	    else { /* I'm not sure if this block is really necessary... */
		unsigned long base0, mem_size;
		unsigned int *mem_base;
		/* Find pdev with pci_find_device() */ 
		base0 = pcimem[i]->base_address[2] & PCI_BASE_ADDRESS_MEM_MASK; 
		mem_size = 0x01000000; 
		mem_base = ioremap(base0, mem_size);
		Cprintk(KERN_INFO "mem_base is %016lx\n", (unsigned long)mem_base);
		Cprintk(KERN_INFO "base2 is %016lx\n", base0);
	    }
	}
	if (i == 0) {
	    printk(KERN_ALERT "no pcimem found\n");
	    return (-ENODEV);
	}
	else {
	    printk(KERN_ALERT "%d pcimem(s) found\n", i);
	    return (i);
	}
    }
    else {
	printk(KERN_ALERT "pci is not supported on this machine\n");
	return (-ENODEV);
    }
}
