/*
 * Host Interface Bridge device driver for GRAPE-7/GRAPE-DR
 * 
 * DMA strategy:
 * + init_module() allocate DMA read&write buf in kernel space
 * + GET_DMAR_PA ioctl returns phys addr of the read buf
 * + GET_DMAW_PA ioctl returns phys addr of the write buf
 * + specify mmap behavior by SET_MAP_MODE ioctl
 * + mmap maps HIB local registers              if mapmode==HIB_MAP_HIBMEM (BAR0)
 *             HIB backend device               if mapmode==HIB_MAP_BACKEND (BAR1)
 *             HIB internal double SRAM buffers if mapmode==HIB_MAP_DBLBUF  (BAR2)
 *             DMA read buf in kernel space      if mapmode==HIB_MAP_DMARBUF (main memory)
 *             DMA write buf in kernel space     if mapmode==HIB_MAP_DMAWBUF (main memory)
 */

#include <asm/current.h>

#if defined(CONFIG_SMP) && !defined(__SMP__)
#define __SMP__
#endif

#include <linux/fs.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/pci.h>
#include <linux/mm.h>
#include <linux/sched.h>
#include <linux/interrupt.h>
#include <linux/version.h>
#ifdef CONFIG_X86_PAT
#include <asm/pat.h>
#include <asm/pci.h>
#endif // CONFIG_X86_PAT
#include <asm/io.h>
//#include <asm/page.h>
#include <asm/pgtable.h>
#include <asm/uaccess.h>
#include "hibdrv.h"

#if 0
#define Cprintk(fmt, args...) printk(fmt, ## args)
#else
#define Cprintk(fmt, args...)
#endif

#ifndef LINUX_VERSION_CODE
#define LINUX_VERSION_CODE 0x020600
#warning LINUX_VERSION_CODE undefined. assume version 0x020600.
#endif

/*
 * vm-operation method to handle page fault.
 */
#if LINUX_VERSION_CODE < 0x02061a // earlier than 2.6.26. use nopage() method.
#define HAS_VMOP_FAULT (0)
#else // 2.6.26 or later. use fault() method.
#define HAS_VMOP_FAULT (1)
#endif

/*
 * kernel interface to initialize a page reference count.
 */
#if LINUX_VERSION_CODE < 0x020613 // earlier than 2.6.19. use get_page().
#define HAS_INIT_PAGE_COUNT (0)
#else // 2.6.13 or later. use init_page_count().
#define HAS_INIT_PAGE_COUNT (1)
#endif

#if LINUX_VERSION_CODE < 0x020400
#warning "older than kernel 2.4"
#elif LINUX_VERSION_CODE < 0x020600
#warning "older than kernel 2.6 but newer than 2.3"
#else 
// #warning "kernel 2.6 or later"
#endif

/* kernel 2.2 or earlier */
#if LINUX_VERSION_CODE < 0x020400
#define pci_resource_start(dev, region) (dev->base_address[region])
#ifndef VMA_OFFSET
#define VMA_OFFSET(vma) (vma->vm_offset)
#endif
#ifndef VM_RESERVED
#define VM_RESERVED (0)
#endif
#define Page unsigned long
int
pci_enable_device(dev)
{
    return (0);
}
static Page
virt_to_page(unsigned long virtual)
{
    return virtual;
}
static void
get_page(Page pageptr)
{
    atomic_inc(&mem_map[MAP_NR(pageptr)].count);
}

#else /* kernel 2.4 or later */

#ifndef VMA_OFFSET
#define VMA_OFFSET(vma) ((vma->vm_pgoff)<<PAGE_SHIFT)
#endif
#define Page struct page*
#endif /* LINUX_VERSION_CODE */

/* file operation methods */
static int hib_open(struct inode *ip, struct file *fp);
static int hib_close(struct inode *ip, struct file *fp);
static int hib_ioctl(struct inode *ip, struct file *fp,
			unsigned int cmd, unsigned long arg);
static int hib_mmap(struct file *fp, struct vm_area_struct *vp);

/* vm operation methods */
#if HAS_VMOP_FAULT
static int hib_dmar_vma_fault(struct vm_area_struct *vma, struct vm_fault *vmf);
static int hib_dmaw_vma_fault(struct vm_area_struct *vma, struct vm_fault *vmf);
#else
static Page hib_dmar_vma_nopage(struct vm_area_struct *vp, unsigned long address, int *type);
static Page hib_dmaw_vma_nopage(struct vm_area_struct *vp, unsigned long address, int *type);
#endif
static void hib_vma_open(struct vm_area_struct *vp);
static void hib_vma_close(struct vm_area_struct *vp);
static void release_pages(int order, struct pci_dev *dev, unsigned long *va, dma_addr_t pa);

/* an interrupt handler */
irqreturn_t hib_intr_handler(int irq, void *dev_id, struct pt_regs *regs);

/* local utility funcs */
static int n_hib_exist(void);

/* local defs */
static int major;
static struct pci_dev *hib[NHIBMAX];
static unsigned long *dmarbuf[NHIBMAX];
static unsigned long *dmawbuf[NHIBMAX];
static dma_addr_t dmarbufpa[NHIBMAX];
static dma_addr_t dmawbufpa[NHIBMAX];
static unsigned int mapmode[NHIBMAX];
static int nhib; /* # of hib devices found */
#if HIB_USE_INTX
static int irq = -1;
#endif
static struct file_operations fops =
{
    .ioctl = hib_ioctl,
    .mmap = hib_mmap,
    .open = hib_open,
    .release = hib_close,
};
static struct vm_operations_struct dmar_vmops =
{
    .open = hib_vma_open,
    .close = hib_vma_close,
#if HAS_VMOP_FAULT
    .fault = hib_dmar_vma_fault,
#else
    .nopage = hib_dmar_vma_nopage,
#endif
};

static struct vm_operations_struct dmaw_vmops =
{
    .open = hib_vma_open,
    .close = hib_vma_close,
#if HAS_VMOP_FAULT
    .fault = hib_dmaw_vma_fault,
#else
    .nopage = hib_dmaw_vma_nopage,
#endif
};

static void
release_pages(int order, struct pci_dev *dev,
	   unsigned long *va, dma_addr_t pa)
{
  int i;
  Page toppg = virt_to_page((unsigned long)va);

  for (i = 1; i < (1 << order); i++) { /* note that i starts from 1, not from 0.
					  reference count of the 1st page is automatically
					  handled by pci_free/alloc_consistent. */
      put_page_testzero(toppg + i);
  }
  pci_free_consistent(dev, PAGE_SIZE * (1<<order), va, pa);
}

int
hib_init(void)
{
    int i, ii;
    Page pageptr;

    /* register this device driver */
    major = register_chrdev(0, HIB_DEVNAME, &fops);

    /* init static vars */
    for (i = 0; i < NHIBMAX; i++) {
	hib[i] = NULL;
	dmarbuf[i] = NULL;
	dmawbuf[i] = NULL;
	mapmode[i] = HIB_MAP_HIBMEM;
    }

    /* look for hib device(s) */
    nhib = n_hib_exist();
    if (!nhib) {
        unregister_chrdev(major, HIB_DEVNAME); 
	return (-ENODEV);
    }

    /* DMA configuration */
    for (i = 0; i < nhib; i++) {
	/* set DMA masks */
	if (pci_set_dma_mask(hib[i], 0xffffffff)) {
	    printk(KERN_WARNING "hib: No suitable DMA available.\n");
            unregister_chrdev(major, HIB_DEVNAME); 
	    return (-ENODEV);
	}
	// printk(KERN_WARNING "hib: DMA mask set to 32-bit.\n");

	if (pci_set_consistent_dma_mask(hib[i], 0xffffffff)) {
	    printk(KERN_WARNING "hib: No suitable consistent DMA available.\n");
            unregister_chrdev(major, HIB_DEVNAME); 
	    return (-ENODEV);
	}
	// printk(KERN_WARNING "hib: consistent DMA mask set to 32-bit.\n");

	/* allocate DMA read buffers */
	dmarbuf[i] = pci_alloc_consistent(hib[i], HIB_DMABUF_BYTES, &dmarbufpa[i]);
	if (NULL == dmarbuf[i]) {
	    printk(KERN_ALERT "hib: pci_alloc_consistent failed\n");
	    for (ii = i-1; ii >= 0; ii--) {
	        release_pages(HIB_DMABUF_ORDER, hib[ii], dmarbuf[ii], dmarbufpa[ii]);
	    }
	    for (ii = i-1; ii >= 0; ii--) {
	        release_pages(HIB_DMABUF_ORDER, hib[ii], dmawbuf[ii], dmawbufpa[ii]);
	    }
            unregister_chrdev(major, HIB_DEVNAME); 
	    return (-ENOMEM);
	}
	pageptr = virt_to_page((unsigned long)dmarbuf[i]);
	for (ii = 1; ii < (1 << HIB_DMABUF_ORDER); ii++) {
#if HAS_INIT_PAGE_COUNT
            init_page_count(pageptr + ii);
#else
            get_page(pageptr + ii);
#endif
	}

	/* allocate DMA write buffers */
	dmawbuf[i] = pci_alloc_consistent(hib[i], HIB_DMABUF_BYTES, &dmawbufpa[i]);
	if (NULL == dmawbuf[i]) {
	    printk(KERN_ALERT "hib: pci_alloc_consistent failed\n");
	    for (ii = i; ii >= 0; ii--) {
	        release_pages(HIB_DMABUF_ORDER, hib[ii], dmarbuf[ii], dmarbufpa[ii]);
	    }
	    for (ii = i-1; ii >= 0; ii--) {
	        release_pages(HIB_DMABUF_ORDER, hib[ii], dmawbuf[ii], dmawbufpa[ii]);
	    }
            unregister_chrdev(major, HIB_DEVNAME); 
	    return (-ENOMEM);
	}
	pageptr = virt_to_page((unsigned long)dmawbuf[i]);
	for (ii = 1; ii < (1 << HIB_DMABUF_ORDER); ii++) {
#if HAS_INIT_PAGE_COUNT
            init_page_count(pageptr + ii);
#else
            get_page(pageptr + ii);
#endif
	}
    }

#if 0
    // just for test
    for (i = 0; i < 4097; i++) {
      dmawbuf[0][i] = 0x12345678abc00000ll | i;
    }
#endif

#if HIB_USE_INTX
    if (irq < 0) {
        // unsigned int data;
        // pci_read_config_dword(hib[0], 0x3c, &data);
        irq = hib[0]->irq & 0x000000ff;
        printk(KERN_ALERT "hib: irq %d\n", irq);
    }
    unsigned long opt_flags = SA_INTERRUPT | SA_SHIRQ;
    int err = request_irq(irq, hib_intr_handler, opt_flags, HIB_DEVNAME, hib);
    if (err) {
        printk(KERN_ALERT "hib: request_irq failed\n");
        return err;
    }
#endif

    printk(KERN_ALERT "hib installed\n");

    return (0);
}

void
hib_exit(void)
{
    int i;

#if HIB_USE_INTX
    //    disable_irq(irq);
    free_irq(irq, hib);
#endif

    for (i = 0; i < nhib; i++) {
        release_pages(HIB_DMABUF_ORDER, hib[i], dmarbuf[i], dmarbufpa[i]);
        release_pages(HIB_DMABUF_ORDER, hib[i], dmawbuf[i], dmawbufpa[i]);
    }

    unregister_chrdev(major, HIB_DEVNAME);
    printk(KERN_ALERT "hib uninstalled\n");
}

module_init(hib_init);
module_exit(hib_exit);
MODULE_LICENSE("Dual BSD/GPL");

static int
hib_open(struct inode *ip, struct file *fp)
{
    Cprintk(KERN_ALERT "hib_open\n");
    /*
    MOD_INC_USE_COUNT;
    */
    return (0);
}

static int
hib_close(struct inode *ip, struct file *fp)
{
    Cprintk(KERN_ALERT "hib_close\n");
    /*
    MOD_DEC_USE_COUNT;
    */
    return (0);
}

static int
hib_ioctl(struct inode *ip, struct file *fp, 
	     unsigned int cmd, unsigned long arg)
{
    int minor = MINOR(ip->i_rdev);
    Cprintk(KERN_ALERT "hib_ioctl cmd: %x\n", (int) cmd);

    /* exec cmd with argument arg */
    switch (cmd) {
      case HIB_WRITE_CFG: {
          struct long_access myarg;
	  if (copy_from_user(&myarg, (struct long_access *)arg, sizeof(myarg))) return (-EFAULT);
	  pci_write_config_dword(hib[minor], myarg.addr, myarg.data);

	  Cprintk(KERN_ALERT "hib_ioctl done WRITE_CFG\n");

	  break;
      }
      case HIB_READ_CFG: {
	  struct long_access myarg;

	  Cprintk(KERN_ALERT "hib_ioctl will READ_CFG\n");
	  if (copy_from_user(&myarg, (struct long_access *)arg, sizeof(myarg))) return (-EFAULT);

	  pci_read_config_dword(hib[minor], myarg.addr, &(myarg.data));
	  Cprintk(KERN_ALERT "hib_ioctl READ_CFG  addr: %x  data: %x\n", myarg.addr, myarg.data);

	  if (copy_to_user((struct long_access *)arg, &myarg, sizeof(myarg))) return (-EFAULT);

	  Cprintk(KERN_ALERT "hib_ioctl done READ_CFG\n");

	  break;

      }
      case HIB_GET_DMAR_PA: {
	  unsigned long baddr;

	  Cprintk(KERN_ALERT "hib_ioctl will GET_DMAR_PA\n");

	  baddr = dmarbufpa[minor];
	  if (copy_to_user((unsigned int *)arg, &baddr, sizeof(baddr))) return (-EFAULT);

	  Cprintk(KERN_ALERT "GET_DMAR_PA: DMA read buf bus addr: %016lx\n", baddr);
	  Cprintk(KERN_ALERT "hib_ioctl done GET_DMAR_PA\n");
	  break;
      }
      case HIB_GET_DMAW_PA: {
	  unsigned long baddr;

	  Cprintk(KERN_ALERT "hib_ioctl will GET_DMAW_PA\n");

	  baddr = dmawbufpa[minor];
	  if (copy_to_user((unsigned int *)arg, &baddr, sizeof(baddr))) return (-EFAULT);

	  Cprintk(KERN_ALERT "GET_DMAW_PA: DMA write buf bus addr: %016lx\n", baddr);
	  Cprintk(KERN_ALERT "hib_ioctl done GET_DMAW_PA\n");
	  break;
      }
      case HIB_SET_MAP_MODE: {
          Cprintk(KERN_ALERT "hib_ioctl will SET_MAP_MODE\n");

	  mapmode[minor] = (unsigned int)arg;
	  Cprintk(KERN_ALERT "SET_MAP_MODE: mapmode: %d\n", mapmode[minor]);

	  Cprintk(KERN_ALERT "hib_ioctl done SET_MAP_MODE\n");

	  break;
      }
      default:
	return (-EINVAL);
    }
    return (0);
}

#if HAS_VMOP_FAULT // implement fault() method.

static int
hib_dmar_vma_fault(struct vm_area_struct *vma, struct vm_fault *vmf)
{
    int minor = MINOR(vma->vm_file->f_dentry->d_inode->i_rdev);
    Page pageptr;
    unsigned long offset = vmf->pgoff << PAGE_SHIFT;
    // address - vma->vm_start + VMA_OFFSET(vma);

    Cprintk(KERN_ALERT "hib_dmar_vma_fault minor: %d\n", minor);
    pageptr = virt_to_page((unsigned long)dmarbuf[minor] + offset);

    vmf->page = pageptr;
    get_page(pageptr);

    return 0;
}

static int
hib_dmaw_vma_fault(struct vm_area_struct *vma, struct vm_fault *vmf)
{
    int minor = MINOR(vma->vm_file->f_dentry->d_inode->i_rdev);
    Page pageptr;
    unsigned long offset = vmf->pgoff << PAGE_SHIFT;
    // address - vma->vm_start + VMA_OFFSET(vma);

    Cprintk(KERN_ALERT "hib_dmaw_vma_fault minor: %d\n", minor);
    pageptr = virt_to_page((unsigned long)dmawbuf[minor] + offset);

    vmf->page = pageptr;
    get_page(pageptr);

    return 0;
}

#else // implement nopage() method.

static Page
hib_dmar_vma_nopage(struct vm_area_struct *vp,
	      unsigned long address, int *type)
{
    int minor = MINOR(vp->vm_file->f_dentry->d_inode->i_rdev);
    Page pageptr;
    unsigned long offset = address - vp->vm_start + VMA_OFFSET(vp);

    Cprintk(KERN_ALERT "hib_dmar_vma_nopage minor: %d\n", minor);
    pageptr = virt_to_page((unsigned long)dmarbuf[minor] + offset);
    get_page(pageptr);

    return pageptr;
}

static Page
hib_dmaw_vma_nopage(struct vm_area_struct *vp,
	      unsigned long address, int *type)
{
    int minor = MINOR(vp->vm_file->f_dentry->d_inode->i_rdev);
    Page pageptr;
    unsigned long offset = address - vp->vm_start + VMA_OFFSET(vp);

    Cprintk(KERN_ALERT "hib_dmaw_vma_nopage minor: %d\n", minor);
    pageptr = virt_to_page((unsigned long)dmawbuf[minor] + offset);
    get_page(pageptr);

    Cprintk(KERN_ALERT "pageptr: 0x%08x  dmawbuf[%d]: 0x%08x  offset: 0x%08x\n",
	   pageptr, minor, dmawbuf[minor], offset);

    return pageptr;
}

#endif

static void
hib_vma_open(struct vm_area_struct *vp)
{
    Cprintk(KERN_ALERT "hib_vma_open\n");
    // MOD_INC_USE_COUNT;
}

static void
hib_vma_close(struct vm_area_struct *vp)
{
    Cprintk(KERN_ALERT "hib_vma_close\n");
    // MOD_DEC_USE_COUNT;
}

static int
hib_mmap(struct file *fp, struct vm_area_struct *vp)
{
    unsigned long off = VMA_OFFSET(vp);
    unsigned long virtual = vp->vm_start;
    unsigned long physical;
    unsigned long vsize = vp->vm_end - vp->vm_start;
    int ret = 0;
    int minor = MINOR(fp->f_dentry->d_inode->i_rdev);
    int bar;

    switch (mapmode[minor]) {
      case HIB_MAP_HIBMEM: /* map HIB local memory */
	bar = 0;
	break;
      case HIB_MAP_BACKEND: /* map HIB backend address space */
	bar = 1;
	break;
      case HIB_MAP_DBLBUF: /* map HIB internal double SRAM buffers */
	bar = 2;
	break;
      default:
	bar = 0;
    }

    switch (mapmode[minor]) {
      case HIB_MAP_HIBMEM: /* map HIB local memory */
      case HIB_MAP_BACKEND: /* map HIB backend address space */
      case HIB_MAP_DBLBUF: /* map HIB internal double SRAM buffers */

	physical  = pci_resource_start(hib[minor], bar) & PCI_BASE_ADDRESS_MEM_MASK;
	/* map BAR to the user space */
	// printk(KERN_ALERT "bar%d physical: 0x%0lx\n", bar, physical);
	physical += off;
	// vp->vm_page_prot = pgprot_noncached(vp->vm_page_prot);
	vp->vm_flags |= VM_RESERVED;

#if LINUX_VERSION_CODE < 0x020503 /* earlier than 2.5.3 */
	ret = remap_page_range(virtual, physical, vsize, vp->vm_page_prot);
#elif LINUX_VERSION_CODE < 0x02060a /* earlier than 2.6.10 */
	ret = remap_page_range(vp, virtual, physical, vsize, vp->vm_page_prot);
#else /* 2.6.10 or later */

        if (mapmode[minor] == HIB_MAP_DBLBUF) {
#ifdef CONFIG_X86_PAT // usr PAT to set BAR2 region as write combined.
            /*
              PTE encoding (see arch/x86/pat.c for detail).
                PAT,PCD,PWT
                  0   0   0    write back
                  0   0   1    write combine
                  0   1   0    uncached-
                  0   1   1    uncached
             */
            // printk(KERN_ALERT " 0) vm_page_prot: 0x%x\n", vp->vm_page_prot);
            // printk(KERN_ALERT "   pat_enabled:%d\n", pat_enabled);
            vp->vm_page_prot =  __pgprot(pgprot_val(vp->vm_page_prot) & ~_PAGE_PAT);
            vp->vm_page_prot =  __pgprot(pgprot_val(vp->vm_page_prot) & ~_PAGE_PCD);
            vp->vm_page_prot =  __pgprot(pgprot_val(vp->vm_page_prot) |  _PAGE_PWT);
            // printk(KERN_ALERT " 1) vm_page_prot: 0x%x\n", vp->vm_page_prot);

#else // use MTRR to set BAR2 region as write combined.
            // unsigned long len = pci_resource_len(grapedrp[minor], bar);
            // int mtrr = mtrr_add(physical, len, MTRR_TYPE_WRCOMB, 1);
            // printk(KERN_ALERT " mtrr: %d\n", mtrr);

#endif // CONFIG_X86_PAT
        }
        ret = remap_pfn_range(vp, virtual, physical >> PAGE_SHIFT, vsize, vp->vm_page_prot);
#endif

	if (ret) return (-EAGAIN);
	/* printk(KERN_ALERT "HIB_MAP_HIBMEM done\n"); */
	break;

    case HIB_MAP_DMARBUF: /* map DMA read buf */
	vp->vm_ops = &dmar_vmops;
	// vp->vm_page_prot = pgprot_noncached(vp->vm_page_prot); // !!!  may be dangerous to remove this line.
	vp->vm_flags |= VM_RESERVED | VM_IO;
	hib_vma_open(vp);
	Cprintk(KERN_ALERT "HIB_MAP_DMARBUF done\n");
	break;

    case HIB_MAP_DMAWBUF: /* map DMA write buf */
	vp->vm_ops = &dmaw_vmops;
        // vp->vm_page_prot = pgprot_noncached(vp->vm_page_prot); // !!!
	vp->vm_flags |= VM_RESERVED | VM_IO;
	hib_vma_open(vp);
	Cprintk(KERN_ALERT "HIB_MAP_DMAWBUF done\n");
	break;

    default:
	printk(KERN_ALERT "unknown map mode %d\n", mapmode[minor]);
	ret = 1;
	break;
    }
    if (ret) {
	return (-EAGAIN);
    }
    return (ret);
}

static int
n_hib_exist(void)
{
    int i, type;
    unsigned int vid, did;
    int ntype = sizeof(hibids) / sizeof(HibId);
    struct pci_dev *hib0;

    i = 0;
    for (type = 0; type < ntype; type++) {
        hib0 = NULL;
        vid = hibids[type].vendorid;
        did = hibids[type].deviceid;
        printk(KERN_ALERT "searching hib with vendorid:0x%04x  deviceid:0x%04x\n", vid, did);
        while (1) {
            hib[i] = pci_get_device(vid, did, hib0);
            hib0 = hib[i];
            if (!hib[i]) { // no more device.
                break;
            }
            else if (pci_enable_device(hib[i])) { // something wrong with the device.
                return (-EIO);
            }
            i++; // found a valid device.
            printk(KERN_ALERT "found a hib with vendorid:0x%04x  deviceid:0x%04x\n", vid, did);
        }
    }
    if (i == 0) {
	printk(KERN_ALERT "no hib found\n");
	return (-ENODEV);
    }
    else if (i == 1) {
	printk(KERN_ALERT "%d hib found\n", i);
	return (i);
    }
    else {
	printk(KERN_ALERT "%d hibs found\n", i);
	return (i);
    }
}


irqreturn_t
hib_intr_handler(int irq, void *dev_id, struct pt_regs *regs)
{
    static int cnt = 0;
#if 0 // irq is shared.
    if () { // handle if this is an interrupt for hibdrv.
        printk(KERN_ALERT "hello %d from hib_intr_handler.\n", cnt++);
        return IRQ_HANDLED;
    }
    else { // otherwise omit it.
        return IRQ_NONE;
    }
#else // irq is not shared.
    printk(KERN_ALERT "hello %d from hib_intr_handler.\n", cnt++);
    return IRQ_HANDLED;
#endif
}
